// garageDoor_Teensy_9x_b6.ino  Ver 1.1 2024/11/04
// Ver 1.1 fixes Radio Reset pin to High-Z when not LOW

// Teensy 3.1 with Feather rf9x wing  Node ID 0xb6
// Closed == 2467, Open 1663,  delta closed to open == 804
// to scale 804 to 255 for display => * 2595 / 2^13 == 254.7 ==> * 2595 >> 13 avoid division which is slow
// i.e. max 804* 2595 == 2088975 so no over flow for uint32_t
// in range 0 to 804 Closed if < 16, Open > 804-16 = 789
// opening >+10 in 0.4sec, after 2sec
// closing <-10 in 0.4sec, after 2sec
// stopped not detected. set from moving + button press OR closed or Opened
/**
   Select Teensy3.2 as the board to compile this sketch
   use Teensy 3.x Feather Adapter and Adafruit LoRa Radio FeatherWing - RFM95W
   On RF9x Feather Wing wire RST to pin A, CS to pin B and IRQ to pin C
   NOTE!! Teensy uses pin 13 as SPI SCK so DO NOT try and drive the Teensy LED
    as that will interfer with the SPI connection to the Radio chip
*/

/**
        motorState transition matrix
   current State  State After button press
   OPENING          STOPPED_OPENING
   CLOSING          STOPPED_CLOSING
   STOPPED_CLOSING  OPENING
   STOPPED_OPENING  CLOSING
*/
/*
   (c)2014-2018 Forward Computing and Control Pty. Ltd.
   NSW Australia, www.forward.com.au
   This code is not warranted to be fit for any purpose. You may only use it at your own risk.
   This code may be freely used for both private and commercial use
   Provided this copyright is maintained.
*/
#include <SPI.h>
#include <RH_RF95.h>
// download the libraries from http://www.forward.com.au/pfod/pfodParserLibraries/index.html
// pfodParser.zip V3.29+ contains pfodParser, pfodSecurity, pfodBLEBufferedSerial, pfodSMS and pfodRadio
#include <pfodSecurity.h>
#include <pfodRadio.h>
#include "VGuageGarageDoor.h"
#include "GarageDoorButton.h"
#include "DebouncedSwitch.h"

// uncomment the following to get debug output to Serial
//#define DEBUG
// uncommnet the next line to send debug output to LoRa connection instead of to USB
//#define DEBUG_SEND

Stream* outputStream; // either &Serial or &plotDataRingBuffer
// uncomment the next line to also check limit switch inputs for open/close
// this will override position sensor
// not used here the limit switches move and bounce
//#define CHECK_LIMIT_SWITCHES

// for convience use last byte of LoRa address as nodeID, but need to check it is unique in this situation.
// the server address for clients to connect to
const uint8_t NODE_ID = 0xb6; // this node's address can be any address from 1 to 0xff,  the server address for clients to connect to

// Change to 434.0 or other frequency, must match the TX's, pfodClient's, freq!  AND must match the range allowed in your country.
#define RF95_FREQ 915.0
// == 81.9mm length wire for 1/4 wave length antenna, see https://www.easycalculation.com/physics/electromagnetism/antenna-wavelength.php


// add your pfod Password here for 128bit security
// eg "b0Ux9akSiwKkwCtcnjTnpWp" but generate your own key, "" means no pfod password
#define pfodSecurityCode "WuxwKgpcEx((UXEqddHstqp"
// see http://www.forward.com.au/pfod/ArduinoWiFi_simple_pfodDevice/index.html for more information and an example
// and QR image key generator.

// ======================
// used to suppress warning
#define pfod_MAYBE_UNUSED(x) (void)(x)

/**
  Radio sends are broken up into blocks of <255 bytes, with measured tx time of 460mS per 255 byte block
  so sends from Server (max 1023 bytes) can take upto 2secs to be received at the client
  Each radio msg is acked. The default ack timeout is 500mS, i.e. random timeout in the range 500mS to 1000mS  (use setAckTimeout(mS) to change this)
  This time allows for another, interfering, radio to finish its attempted transmission
  After 5 retries the link is marked as failed. (use setNoOfRetries() to change this number)
  Most pfod commands are very short, <20 bytes (max 254 bytes), but responses from the pfodServer (pfodDevice) can be upto 1023 bytes.
  Since only one radio can transmit at a time, you should only send RawData from the pfodServer while the pfodClient is waiting for a response
  That is only send RawData when the server receives a command and before it sends its response.
*/

//==================================== pfodRHDriver ================
// The class to interface with the low lever radio driver.  Must extend from pfodRadioDrive interface
// class implementation is at the bottom of this file.
class pfodRHDriver : public pfodRadioDriver {
  public:
    pfodRHDriver(RHGenericDriver* _driver);  // RHGenericDriver is the low lever RadioHead class used to talk to the radio module
    bool init();  int16_t lastRssi();   int getMode();   uint8_t getMaxMessageLength();   void setThisAddress(uint8_t addr);
    bool receive(uint8_t* buf, uint8_t* len);   bool send(const uint8_t* data, uint8_t len);   uint8_t headerTo();
    uint8_t headerFrom();   uint8_t headerId();   uint8_t headerFlags(); void setHeaderTo(uint8_t to);
    void setHeaderFrom(uint8_t from);   void setHeaderId(uint8_t id);  void setHeaderFlags(uint8_t flags);
  private:
    RHGenericDriver* driver;
};
//============= end of pfodRHDriver header

/* Teensy 3.x w/wing */
// on RF9x wing wire RST to A, CS to B and IRQ to C
#define RFM95_RST 9 // "A"
#define RFM95_CS 10 // "B"
#define RFM95_INT 4 // "C"
// MISO, SCK, MOSI hardwired to the correct pins on the Radio feather wing
// MOSI: pin 11   MISO: pin 12   SCK: pin 13  <<<< NOTE DO NOT DRIVE THE LED!!

// Singleton instance of the low level radio driver
RH_RF95 driver(RFM95_CS, RFM95_INT);

pfodRHDriver pfodDriver(&driver); // create the pfodRadioDriver class

pfodRadio radio(&pfodDriver, NODE_ID); // the high level pfod radio driver class
pfodSecurity parser("V3"); // create a parser to handle the pfod messages
pfodDwgs dwgs(&parser);  // drawing support
// Commands for drawing controls only need to unique per drawing, you can re-use cmds 'a' in another drawing.

// set up a ring byte buffer to buffer data points
// as can only safely send data after we have recieved a cmd and
// pfodApp is waiting for the response
// It would use less RAM if we just stored the data points
// but pfod library already has a byte ring buffer
// and there is RAM to spare on Feather M0, 32K RAM
pfodRingBuffer plotDataRingBuffer;
// note pfodRadio only as 1024 byte raw data send buffer which discards extra bytes when full
const size_t plotDataBufferSize = 1023;
uint8_t plotDataBuffer[plotDataBufferSize];
void bufferPlotData(); // method prototype
void sendRawData(); // method prototype


const unsigned long ADC_INTERVAL = 3; // 3mS == 0.192sec filter length
unsigned long ADCtimerStart = 0;
const int ADC_PIN = 0; // PIN for adc

const int FILTER_SIZE_POWER = 6;
const size_t FILTER_SIZE = (1 << FILTER_SIZE_POWER) - 1;
uint16_t positionFilter[FILTER_SIZE + 1]; // storage for filter
unsigned long filterSum; // current sum
size_t filterIdx = 0; // index into filter
int filteredValue = 0; // the current filter output


// set pin for controlling reed relay
const int relayPin =  6;
unsigned long pulseStartTime = 0;
// the time when we started the relay pulse
bool pulseTimerRunning = false; // set to true when we have a timer running
unsigned long RELAY_PULSE_LENGTH = 100; // 0.1sec keep all timing in unsigned long

// note is change these value need to also change doorGuagePos() calculations to match
uint32_t CLOSED_POSITION = 2467;
uint32_t OPEN_POSITION = 1663;
uint32_t POSITION_RANGE = CLOSED_POSITION - OPEN_POSITION; // == 804

// in range 0 to 804 Closed if < 16, Open > 804-16 = 789
// opening >+10 in 0.4sec, after 2sec
// closing <-10 in 0.4sec, after 2sec
// stopped not detected. set from moving + button press OR closed or Opened
uint32_t closedPositionLimit = 16;
uint32_t openPositionLimit = POSITION_RANGE - closedPositionLimit;
int32_t movementPositionLimit = 6; // need about 1.2 to 1.6sec to get movement detection
int32_t movementDelta = 0;

unsigned long movementStartTime = 0;
unsigned long MOVEMENT_TIME = 400; // add delta position difference over 0.4 sec (4x0.1mS)

unsigned long interPulseMovementStartTime = 0;
// the time when we started the inter pulse timer
bool interPulseMovementTimerRunning = false; // set to true when we have a timer running
unsigned long INTER_PULSE_MOVEMENT_LENGTH = 1700; // 1.7sec keep all timing in unsigned long

unsigned long interPulseStartTime = 0;
// the time when we started the inter pulse timer
bool interPulseTimerRunning = false; // set to true when we have a timer running
unsigned long INTER_PULSE_LENGTH = INTER_PULSE_MOVEMENT_LENGTH + MOVEMENT_TIME + 10; // ~2.1sec 
// need at least one movement measurement before interPulseTimer times out and triggers relay again 
// but want to ignore the first 1.7sec (INTER_PULSE_MOVEMENT_LENGTH) 
// when starting from closed/open to give the door time to get moving


// manual override push button
const int pushButtonPin = 5;
DebouncedSwitch pushButtonSw(pushButtonPin); // monitor a switch on input D5
// this constructor also set the pin as INPUT_PULLUP and captures its initial state

// limit switch pins  // not used here
const int CLOSED_LIMIT_PIN = 16;  // == A2 green wire
const int OPENED_LIMIT_PIN = 17;  // == A3 blue wire
bool closedLimitTripped = false;
bool openLimitTripped = false;

enum MotorStateEnum {OPENING, CLOSING, STOPPED_CLOSING, STOPPED_OPENING };
// motorState CLOSED and OPEN set by either the limit switches OR by the position < 2, > 253 (i.e. 1% from end points)
// CLOSED, OPEN are STOPPED_CLOSING, STOPPED_OPENING + limit switch or position
// also motor state

uint32_t doorPosition = 0; // 0 == closed, 804 == open
uint32_t previousDoorPosition = 0; // 0 == closed, 804 == open
uint32_t guagePosition = 0; // display of door position 0 to 255.  Used for display only

char lastRequestedState = ' '; // start with no request
// lastRequestedState is cleared to ' ' once the door has reached that state

// if going in the correct direction then 1 press to stop then 1 press to start and if then going in the wrong direction then need 1 press to stop and 1 press to go in correct direction
// so max 4 button presses for any command
// note delays in identifying the current door state can result in an initial button press when already in the correct state
int buttonPressCount = 0; // clear on each new request
int maxButtonPressCount = 0; // set on each new request set to zero to inhibit more presses if door jams
const int MAX_BUTTON_PRESS_COUNT = 4; // max number of button presses to achieve any cmd

unsigned long previous_mS;
MotorStateEnum motorState = STOPPED_CLOSING;
MotorStateEnum lastMotorState = STOPPED_CLOSING;

VGuageGarageDoor vguage(&dwgs);
GarageDoorButton openButton('O', &dwgs);
GarageDoorButton stopButton('S', &dwgs);
GarageDoorButton closeButton('C', &dwgs);


// 0 == closed, 804 == open
uint32_t doorPositionCalc(int filteredADC) {
  int32_t offsetPos = filteredADC - OPEN_POSITION; // 0 to 804 for open to closed
  int32_t pos = -(offsetPos - POSITION_RANGE); // 804 to 0 for open to closed
  if (pos < 0) {
    pos = 0;
  }
  if (((uint32_t)pos) > POSITION_RANGE) {
    pos = POSITION_RANGE;
  }
  return pos;
}

// to scale 804 to 255 for display => * 2595 / 2^13 == 254.7 ==> * 2595 >> 13 avoid division which is slow
// i.e. max 804* 2595 == 2088975 so no over flow for uint32_t
int doorGuagePos(uint32_t doorPos) {
  uint32_t multiplyResult = (doorPos * 2595) ;
  uint32_t divResult = multiplyResult >> 13  ;
  if (divResult > 255) {
    divResult = 255;
  }
  return (int)divResult;
}




bool isDoorOpen() {
  return (motorState != CLOSING) && ((doorPosition > openPositionLimit) || (openLimitTripped));
}
bool isDoorClosed() {
  return (motorState != OPENING) && ((doorPosition < closedPositionLimit) || (closedLimitTripped));
}


// uses global doorPosition 0 to 804
// NOTE this timer is < the inter pulse timeout.
void setDoorStateFromPosition() {
#ifdef CHECK_LIMIT_SWITCHES
  // these bounce quite a bit
  checkOpenedClosedLimitSwitches(); // sets closedLimitTripped, openLimitTripped
#endif
  unsigned long currentTime = millis();
  // update difference from 0.4 sec ago
  if ((currentTime - movementStartTime) > MOVEMENT_TIME) {
    movementStartTime = currentTime; // reset timer
    movementDelta = doorPosition - previousDoorPosition;
    previousDoorPosition = doorPosition;
#ifdef DEBUG
    outputStream->print(" delta:"); outputStream->print(movementDelta);
#endif
    if ((interPulseMovementTimerRunning) || (pulseTimerRunning)) {
      // skip this one for now
#ifdef DEBUG
      outputStream->println("  Skip");
#endif
    } else {
#ifdef DEBUG
      outputStream->println();
#endif
      if (movementDelta > movementPositionLimit) {
#ifdef DEBUG
        if (motorState != OPENING) {
          outputStream->print(" state:"); outputStream->println(stateEnumToString(motorState));
        }
#endif
        motorState = OPENING;
      } else if (movementDelta < -movementPositionLimit) {
#ifdef DEBUG
        if (motorState != CLOSING) {
          outputStream->print(" state:"); outputStream->println(stateEnumToString(motorState));
        }
#endif
        motorState = CLOSING;
      } else {
        // stopped or stopping or starting
      }
    }
  }
  if (isDoorOpen()) {
    if (motorState != STOPPED_OPENING) {
#ifdef DEBUG
      outputStream->print(" state:"); outputStream->println(stateEnumToString(motorState));
#endif
      startInterPulseTimer();
    }
    motorState = STOPPED_OPENING;
  }
  if (isDoorClosed()) {
    if (motorState != STOPPED_CLOSING) {
#ifdef DEBUG
      outputStream->print(" state:"); outputStream->println(stateEnumToString(motorState));
#endif
      startInterPulseTimer();
    }
    motorState = STOPPED_CLOSING;
  }

}

//https://stackoverflow.com/questions/4836534/returning-a-pointer-to-a-literal-or-constant-character-array-string
//String literals are guaranteed to live for the span of the application. It might not be in read-only/immutable memory,
//but they're guaranteed to exist for the lifetime of the application, so returning a pointer to one is fine
const char *stateEnumToString(MotorStateEnum state) {
  switch (state) {
    case STOPPED_OPENING:
      if (isDoorOpen()) {
        vguage.setLabel(F("<g><b>Open"));
        return "OPEN";
      } else {
        vguage.setLabel(F("Stopped"));
        return "STOPPED_Opening";
      }
    case STOPPED_CLOSING:
      if (isDoorClosed()) {
        vguage.setLabel(F("Closed"));
        return "CLOSED";
      } else {
        vguage.setLabel(F("Stopped"));
        return "STOPPED_Closing";
      }
    case OPENING:
      if (isDoorOpen()) {
        vguage.setLabel(F("<g><b>Open"));
        return "OPEN";
      } else {
        vguage.setLabel(F("Opening"));
        return "OPENING";
      }
    case CLOSING:
      if (isDoorClosed()) {
        vguage.setLabel(F("Closed"));
        return "CLOSED";
      } else {
        vguage.setLabel(F("Closing"));
        return "CLOSING";
      }
    default:
      vguage.setLabel(F(" "));
      return "unknownDoorState";
  }
}

// print out state every time door state changes
// or every DEBUG_PRINT_INTERVAL
#ifdef DEBUG
void debugPrint(Stream* debugStream) {
  unsigned long currentTime = millis();
  if (lastMotorState != motorState) {
    lastMotorState = motorState;
    debugStream->print(currentTime); debugStream->print(" mS  "); debugStream->print(" ReqState:"); debugStream->print(lastRequestedState);
    debugStream->print(" door:"); debugStream->print(doorPosition); debugStream->print(" guage:"); debugStream->println(guagePosition);
    debugStream->print(" openLimit:"); debugStream->print(openLimitTripped ? "true" : "false");  debugStream->print(" closedLimit:"); debugStream->println(closedLimitTripped ? "true" : "false");
    debugStream->print(" state:"); debugStream->print(stateEnumToString(motorState)); debugStream->print(" delta:"); debugStream->println(movementDelta);
    debugStream->println();
  }
}
#endif


void initializeFilter() {
  for (size_t i = 0; i <= FILTER_SIZE; i++) {
    positionFilter[i] = 0;
  }
  filterSum = 0;
  filterIdx = 0;
  filteredValue = 0;
}


uint32_t filterReading(int adcReading) {
  // get current reading in this idx
  uint32_t idxReading = positionFilter[filterIdx];
  // update sum
  filterSum += adcReading;
  filterSum -= idxReading;
  positionFilter[filterIdx] = adcReading;
  filterIdx++;
  filterIdx &= FILTER_SIZE; // loop if needed
  filteredValue = filterSum >> FILTER_SIZE_POWER; // divide by filter size a power of 2
  // limit to 4095
  filteredValue &= 4095;
  return filteredValue;
}


// sets global doorPosition
void readFilterPosition() {
  unsigned long currentTime = millis();
  if ((currentTime - ADCtimerStart) > ADC_INTERVAL) {
    ADCtimerStart = currentTime;
    int i = analogRead(ADC_PIN);
    int filteredADC = filterReading(i);
    // scale to 0 to 255 later
    // for now leave in range 2
    doorPosition = doorPositionCalc(filteredADC);
    guagePosition = doorGuagePos(doorPosition);
    vguage.setValue(guagePosition);
  }
}

bool previousOpenLimit;
bool previousClosedLimit;

// Note could use debouce for these also but any closure is enough
void checkOpenedClosedLimitSwitches() {
  openLimitTripped = (digitalRead(OPENED_LIMIT_PIN) == LOW);
  closedLimitTripped = (digitalRead(CLOSED_LIMIT_PIN) == LOW);
  if (openLimitTripped && closedLimitTripped) { // invalid
    openLimitTripped = false;
    closedLimitTripped = false;
  }
  if (openLimitTripped != previousOpenLimit) {
    previousOpenLimit = openLimitTripped;
#ifdef DEBUG
    outputStream->print("OpenLimit:");
    outputStream->println(openLimitTripped ? "tripped" : "open");
#endif
  }
  if (closedLimitTripped != previousClosedLimit) {
    previousClosedLimit = closedLimitTripped;
#ifdef DEBUG
    outputStream->print("ClosedLimit:");
    outputStream->println(closedLimitTripped ? "tripped" : "open");
#endif
  }
}

/**
        motorState transition matrix
   current State  State After button press
   OPENING          STOPPED_OPENING
   CLOSING          STOPPED_CLOSING
   STOPPED_CLOSING  OPENING
   STOPPED_OPENING  CLOSING
*/
MotorStateEnum getNewMotorState(MotorStateEnum currentState) {
  MotorStateEnum rtn = currentState;
  switch (currentState) {
    case OPENING:
      rtn = STOPPED_OPENING;
      break;
    case CLOSING:
      rtn = STOPPED_CLOSING;
      break;
    case STOPPED_CLOSING:
      rtn = OPENING;
      break;
    case STOPPED_OPENING:
      rtn = CLOSING;
      break;
  }
#ifdef DEBUG
  if (rtn != currentState) {
    outputStream->print(" state was:"); outputStream->print(stateEnumToString(currentState));
    outputStream->print(" state now:"); outputStream->println(stateEnumToString(rtn));
  }
#endif
  return rtn;
}

// returns true if button pushed
// if this method return true, it does not process motorState for at least 0.1+0.9 = 1sec
// so have time for motorState to be updated by setDoorStateFromPosition as door moves or stops.
bool processButton() {
  if ((interPulseTimerRunning) || (pulseTimerRunning)) {
    // skip this one for now
    return false; // did not push button this time
  }

  // need to add check for stuck door
  // i.e. limit the max number of button presses to achieve request
  // if going in the correct direction then 1 press to stop then 1 press to start and if then going in the wrong direction then need 1 press to stop and 1 press to go in correct direction
  // so max 4 button presses for any command
  // note delays in identifying the current door state can result in an initial button press when already in the correct state

  // check motorState against request if any
  if ((lastRequestedState == 'O') && isDoorOpen()) {
#ifdef DEBUG
    outputStream->println("Reached OPEN");
#endif
#ifdef DEBUG
  if (motorState != STOPPED_OPENING) {
    outputStream->print(" state:"); outputStream->println(stateEnumToString(motorState));
  }
#endif
    motorState = STOPPED_OPENING;
    lastRequestedState = ' '; // finished
    startInterPulseTimer(); // delay here until settles
  } else if ((lastRequestedState == 'C') && isDoorClosed()) {
#ifdef DEBUG
    outputStream->println("Reached CLOSED");
#endif
#ifdef DEBUG
  if (motorState != STOPPED_CLOSING) {
    outputStream->print(" state:"); outputStream->println(stateEnumToString(motorState));
  }
#endif
    motorState = STOPPED_CLOSING;
    lastRequestedState = ' ';
    startInterPulseTimer(); // delay here until settles
  } else if ((lastRequestedState == 'S') && ((motorState == STOPPED_CLOSING) || (motorState == STOPPED_OPENING))) {
#ifdef DEBUG
    outputStream->println("Reached STOPPED");
#endif
    lastRequestedState = ' ';
    startInterPulseTimer(); // delay here until settles
  }

  // check if nothing to do
  // do check here as lastRequestedState may have been cleared above
  if (lastRequestedState == ' ') {
    return false; // nothing to do
  }

  // reach here is last requested state does not match current door state
  if ((lastRequestedState == 'O') && (motorState == OPENING)) {
    return false; // will open eventually
  } else if ((lastRequestedState == 'C') && (motorState == CLOSING)) {
    return false; // will close eventually
  } else if (lastRequestedState == 'S')  {
    // if motorState was already stopped then would have returned above so need to continue here
  }

  buttonPressCount++;
  if (buttonPressCount <= maxButtonPressCount) {
    startRelayPulse(); //push button
    motorState = getNewMotorState(motorState);
    return true; // button pushed
  } else {
#ifdef DEBUG
    outputStream->println(" !!Pulse blocked!!");
#endif
    lastRequestedState = ' '; // clear last cmd
    startInterPulseTimer();
    return false; // button Not pushed
  }
}

void manualResetRadio() {
  // manual radio reset
  pinMode(RFM95_RST, OUTPUT);
  digitalWrite(RFM95_RST, LOW);
  delayMicroseconds(100);
  pinMode(RFM95_RST, INPUT);
  delay(100); // wait 0.1 sec
}

void setup() {
  pinMode(relayPin, OUTPUT); // output for relay
  digitalWrite(relayPin, LOW); // set output

  // initialize ringBuffer to buffer plot data
  plotDataRingBuffer.init(plotDataBuffer, plotDataBufferSize); // initialize ring byte buffer

#ifdef DEBUG
#ifdef DEBUG_SEND
  outputStream = &plotDataRingBuffer;
#else
  Serial.begin(115200);
  delay(100);
  for (int i = 10; i > 0; i--) {
    Serial.print(i); Serial.print(' ');
    delay(500);
  }
  Serial.println("Garage Door LoRa 0x8e");
  outputStream = &Serial;
#endif
  //radio.setDebugStream(&Serial); // need to enable DEBUG in pfodRadio.cpp file as well
  //parser.setDebugStream(&Serial); // need to enable DEBUG in pfodSecurity.cpp file as well
#endif

  driver.setPromiscuous(false); // only accept our address from low level driver
  // set spi select pin and drive high, deselect
  pinMode(RFM95_CS, OUTPUT);
  digitalWrite(RFM95_CS, HIGH);
  bool radioInitialized = false;
  bool freqChanged = false;
  bool configSet = false;
  while ((!radioInitialized) || (!freqChanged) || (!configSet)) {  // loop here until radio inits
    digitalWrite(RFM95_CS, HIGH); // deselect
//    hardware_spi.end();
    manualResetRadio(); // delays 0.1sec after  
    radioInitialized = radio.init();
    if (!radioInitialized) {
#ifdef DEBUG
      outputStream->println("LoRa radio init failed");
#endif
      delay(1000); // pause and then try again
      continue;
    }
    // setup LoRa radio
    // Defaults after init are 434.0MHz, modulation Bw125Cr45Sf128, +13dbM
    freqChanged = driver.setFrequency(RF95_FREQ);
    if (!freqChanged) {
#ifdef DEBUG
      Serial.println("LoRa radio freq set failed");
#endif
      delay(1000); // pause and then try again
      continue;
    }
    configSet = driver.setModemConfig(RH_RF95::Bw125Cr45Sf128); // default
    if (!configSet) {
#ifdef DEBUG
      Serial.println("LoRa radio modem config set failed");
#endif
      delay(1000); // pause and then try again
      continue;
    }
  }
#ifdef DEBUG
  outputStream->println("LoRa radio init OK!");
#endif

  driver.setTxPower(23, false);  // set max Tx power
  radio.listen(); // set as server listening for msg address to NODE_ID
  parser.connect(&radio, F(pfodSecurityCode)); // connect the parser to the i/o stream

  // set some initial values. loop() below will update according to the door position
  lastRequestedState = 'C';
  doorPosition = 0;
  //  previousDoorPosition = 0;
  lastRequestedState = 'C'; // start closed
  previous_mS = millis();

  vguage.setLabel(F("Closed")); // <g>Open, Opening, Closing, Stopped
  vguage.setValue(0);
  openButton.setLabel(F("Open"));
  stopButton.setLabel(F("<r>Stop"));
  closeButton.setLabel(F("Close"));

  pinMode(OPENED_LIMIT_PIN, INPUT_PULLUP);
  pinMode(CLOSED_LIMIT_PIN, INPUT_PULLUP);

  //analogReference(AR_DEFAULT); //3.3V  Feather M0 and teensy default is 3.3V VCC
  analogReadRes(12); // for teensy 3.2
  //analogReadResolution(12); // 12 bit 0 to 4095 for Feather M0

  initializeFilter();
  ADCtimerStart = millis();
}

void sendRawData() {
  // send all the stored data bytes now
  if (plotDataRingBuffer.available()) { // new raw data to send
    while (plotDataRingBuffer.available()) {
      parser.write(plotDataRingBuffer.read());
    }
    parser.flush(); // send the data now
  }
}

//  checkManualOverride(); // has user pressed or relased the manual pushbotton
void checkManualOverride() {
  pushButtonSw.update(); // handle debouce
  if (pushButtonSw.isChanged()) {
    // clear out remote control
    lastRequestedState = ' '; // finished
    interPulseTimerRunning = false;
    pulseTimerRunning = false;
    if (pushButtonSw.isDown()) {
      // user just pressed it
      // activate relay
      digitalWrite(relayPin, HIGH);
#ifdef DEBUG
      outputStream->println("++Relay");
#endif

    } else {
      // release relay
      digitalWrite(relayPin, LOW);
#ifdef DEBUG
      outputStream->println("--Relay");
#endif
    }
  }
}

void loop() {
  readFilterPosition(); // read ADC and filter position, calculate doorPostion (0 to 804) and display guage position (0 to 255)
  controlRelay(); // turns off relay pulse after 0.1sec
  checkInterPulseTimer(); // check for interpulse timeout 0.9sec
  setDoorStateFromPosition(); //set door state enum from position
  stateEnumToString(motorState); // updates guage label
  checkManualOverride(); // has user pressed or relased the manual pushbotton

#ifdef DEBUG
  debugPrint(outputStream);
#endif

  processButton(); // will push button again if lastRequestedState does not match door state
  // and door not going in the requested direction

  byte cmd = parser.parse(); // parse incoming data from connection
  // parser returns non-zero when a pfod command is fully parsed
  if (cmd != 0) { // have parsed a complete msg { to }
    // can safely send raw data here
    sendRawData();

    byte* pfodFirstArg = parser.getFirstArg(); // may point to \0 if no arguments in this msg.
    pfod_MAYBE_UNUSED(pfodFirstArg); // may not be used, just suppress warning
    long pfodLongRtn; // used for parsing long return arguments, if any
    pfod_MAYBE_UNUSED(pfodLongRtn); // may not be used, just suppress warning
    if ('.' == cmd) {
      // pfodApp has connected and sent {.} , it is asking for the main menu
      if (!parser.isRefresh()) {
        sendMainMenu(); // send back the menu designed
      } else {
        sendMainMenuUpdate(); // menu is cached just send update
      }

      // now handle commands returned from dwg's buttons
    } else if ('A' == cmd) {
      byte dwgCmd = parser.parseDwgCmd();  // parse rest of dwgCmd, return first char of active cmd
      if ('O' == dwgCmd) {
#ifdef DEBUG
        outputStream->println("Dwg O cmd");
#endif
        buttonPressCount = 0;
        maxButtonPressCount = MAX_BUTTON_PRESS_COUNT;
        // in the main Menu of Menu_1
        lastRequestedState = 'O';
        // always send back a response or pfodApp will timeout
      } else if ('S' == dwgCmd) {
#ifdef DEBUG
        outputStream->println("Dwg S cmd");
#endif
        buttonPressCount = 0;
        maxButtonPressCount = MAX_BUTTON_PRESS_COUNT;
        // in the main Menu of Menu_1
        lastRequestedState = 'S';
      } else if ('C' == dwgCmd) {
#ifdef DEBUG
        outputStream->println("Dwg C cmd");
#endif
        buttonPressCount = 0;
        maxButtonPressCount = MAX_BUTTON_PRESS_COUNT;
        // in the main Menu of Menu_1
        lastRequestedState = 'C';
      }
      sendDrawingUpdates_z(); // update the drawing
      // always send back a response or pfodApp will timeout


    } else if ('z' == cmd) { // handle dwg load and refresh requests
      // get starting offset
      if (!parser.isRefresh()) {
        parser.parseLong(pfodFirstArg, &pfodLongRtn); // get starting offset
        long startingOffset = pfodLongRtn;
        if (startingOffset == 0) {
          sendDrawing_z0();
        } else {
          parser.print(F("{}")); // finished
        }

      } else {
        sendDrawingUpdates_z();
      }
    } else if ('!' == cmd) {
      // CloseConnection command
      closeConnection(parser.getPfodAppStream());
    } else {
      // unknown command
      parser.print(F("{}")); // always send back a pfod msg otherwise pfodApp will disconnect.
    }
  }

}

void closeConnection(Stream *io) {
  // add any special code here to force connection to be dropped
}

// this is called on receiving
// a command to open/close
// sets up time and turns relay on
void startInterPulseTimer() {
  interPulseStartTime = millis();
  interPulseTimerRunning = true;
  interPulseMovementStartTime = millis();
  interPulseMovementTimerRunning = true;
#ifdef DEBUG
  outputStream->println("+InterDelay");
  outputStream->println("+InterMovementDelay");
#endif
}

void checkInterPulseTimer() {
  if (interPulseTimerRunning && ((millis() - interPulseStartTime) > INTER_PULSE_LENGTH)) {
    interPulseTimerRunning = false; // timer finished
#ifdef DEBUG
    outputStream->println("-InterDelay");
#endif
  }
  if (interPulseMovementTimerRunning && ((millis() - interPulseMovementStartTime) > INTER_PULSE_MOVEMENT_LENGTH)) {
    interPulseMovementTimerRunning = false; // timer finished
#ifdef DEBUG
    outputStream->println("-InterMovementDelay");
#endif
  }
}

// this is called on receiving
// a command to open/close
// sets up time and turns relay on
void startRelayPulse() {
  pulseStartTime = millis();
  pulseTimerRunning = true;
  // make pin high
  digitalWrite(relayPin, HIGH);
#ifdef DEBUG
  outputStream->println("+Relay");
#endif
}

// this is called each loop
// turns relay off once time
// exceeds stopTime
void controlRelay() {
  if (pulseTimerRunning && ((millis() - pulseStartTime) > RELAY_PULSE_LENGTH)) {
    pulseTimerRunning = false; // timer finished
    // make pin low
    digitalWrite(relayPin, LOW);
#ifdef DEBUG
    outputStream->println("-Relay");
#endif
    startInterPulseTimer();
  }
}

void sendMainMenu() {
  // !! Remember to change the parser version string
  //    every time you edit this method
  parser.print(F("{,"));  // start a Menu screen pfod message
  // send menu background, format, prompt, refresh and version
  parser.print(F("~<+5>Garage Door"));
  parser.sendRefreshAndVersion(0); // no auto re-request, and send the menu version
  // send menu items
  parser.print(F("|+A~z")); // only one item the dwg showing the controls
  parser.print(F("}"));  // close pfod message
}

void sendMainMenuUpdate() {
  parser.print(F("{;"));  // start an Update Menu pfod message
  // send menu items
  parser.print(F("|A"));
  parser.print(F("}"));  // close pfod message
  // ============ end of menu ===========
}

void sendDrawing_z0() {
  dwgs.start(50, 50, dwgs.WHITE); // background defaults to WHITE if omitted i.e. response +=
  parser.sendRefreshAndVersion(100); // re-request dwg every 1/10's sec. this is ignored if no parser version set
  dwgs.pushZero(35, 8, 1.5); // move zero to centre of dwg to 25,12 and scale by 1.5 times
  openButton.draw(); // draw the control
  dwgs.popZero();
  dwgs.pushZero(35, 24, 1.85); // move zero to centre of dwg to 25,12 and scale by 1.5 times
  stopButton.draw(); // draw the control
  dwgs.popZero();
  dwgs.pushZero(35, 41, 1.5); // move zero to centre of dwg to 25,12 and scale by 1.5 times
  closeButton.draw(); // draw the control
  dwgs.popZero();
  dwgs.pushZero(12, 38); // move zero to centre of dwg to 9,30 and scale is 1 (default)
  vguage.draw(); // draw the control
  dwgs.popZero();
  dwgs.end();
}

void sendDrawingUpdates_z() {
  dwgs.startUpdate();
  openButton.update(); // update with current state
  stopButton.update(); // update with current state
  closeButton.update(); // update with current state
  vguage.update(); // update with current state
  dwgs.end();
}


// implementation of pfodRHDriver
pfodRHDriver::pfodRHDriver(RHGenericDriver* _driver) {
  driver = _driver;
}
bool pfodRHDriver::init() {
  return driver->init();
}
int16_t pfodRHDriver::lastRssi() {
  return driver->lastRssi();
}
int pfodRHDriver::getMode() {
  return driver->mode();
}
uint8_t pfodRHDriver::getMaxMessageLength() {
  return driver->maxMessageLength(); // it is important this is set
}
void pfodRHDriver::setThisAddress(uint8_t addr) {
  driver->setThisAddress(addr);
}
bool pfodRHDriver::receive(uint8_t* buf, uint8_t* len) {
  return driver->recv(buf, len);
}
bool pfodRHDriver::send(const uint8_t* data, uint8_t len) {
  return driver->send(data, len);
}
uint8_t pfodRHDriver::headerTo() {
  return driver->headerTo();
}
uint8_t pfodRHDriver::headerFrom() {
  return driver->headerFrom();
}
uint8_t pfodRHDriver::headerId() {
  return driver->headerId();
}
uint8_t pfodRHDriver::headerFlags() {
  return driver->headerFlags();
}
void pfodRHDriver::setHeaderTo(uint8_t to) {
  driver->setHeaderTo(to);
}
void pfodRHDriver::setHeaderFrom(uint8_t from) {
  driver->setHeaderFrom(from);
}
void pfodRHDriver::setHeaderId(uint8_t id) {
  driver->setHeaderId(id);
}
void pfodRHDriver::setHeaderFlags(uint8_t flags) {
  driver->setHeaderFlags(flags, 0xff);
}
// end of pfodRHDriver implementation
