/*
 pfodParserMicroBit library V1.0   20th October 2016 (www.pfod.com.au)
 (c)2016 Forward Computing and Control Pty. Ltd.
 This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 This code may be freely used for both private and commercial use.
 Provide this copyright is maintained.
 */


/*
This code derived from BBC micro:bit C++ MicroBitUARTService.cpp used under the following licence
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
 */

/**
 * Class definition for the custom MicroBit UART Service.
 * Provides a BLE service that acts as a UART port, enabling the reception and transmission
 * of an arbitrary number of bytes.
 */

#include "ble/UUID.h"

#include "ExternalEvents.h"
#include "pfodBleUart.h"
#include "MicroBitFiber.h"
#include "ErrorNo.h"
#include "NotifyEvents.h"

//#define UNUSED(x) (void)(x)

static uint8_t txBufferHead = 0;
static uint8_t txBufferTail = 0;

static GattCharacteristic* txCharacteristic = NULL;

/**
 * A callback function for whenever a Bluetooth device consumes our TX Buffer
 */
void pfod_on_confirmation(uint16_t handle) {
    if (handle == txCharacteristic->getValueAttribute().getHandle()) {
        txBufferTail = txBufferHead;
        MicroBitEvent(PFOD_ID_UART, PFOD_UART_DATA_SENT);
    }
}

/**
 * Constructor for the UARTService.
 * @param _ble an instance of BLEDevice
 * @param rxBufferSize the size of the rxBuffer
 * txBufferSize the size is fixed at 20 bytes
 *
 */
pfodBleUart::pfodBleUart(BLEDevice &_ble, int rxBufSize) : ble(_ble) {
    //    rxBufferHead = 0;
    //    rxBufferTail = 0;
    //    txBufferHead = 0;
    //    txBufferTail = 0;
    //    pfodTxBufferIdx = 0;
    init(); // clears buffers
    txBufferSize = PFOD_UART_DEFAULT_TX_BUF_SIZE;
    rxBufferSize = rxBufSize;
    rxBufferSize += 1;
    txBufferSize += 1;

    txBuffer = (uint8_t *) malloc(txBufferSize);
    rxBuffer = (uint8_t *) malloc(rxBufferSize);

    this->rxBufferSize = rxBufferSize;

    this->txBufferSize = txBufferSize;

    GattCharacteristic rxCharacteristic(UARTServiceRXCharacteristicUUID, rxBuffer, 1, rxBufferSize, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);

    txCharacteristic = new GattCharacteristic(UARTServiceTXCharacteristicUUID, txBuffer, 1, txBufferSize, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_INDICATE); //); //BLE_GATT_CHAR_PROPERTIES_NOTIFY does not notify

    GattCharacteristic * charTable[] = {txCharacteristic, &rxCharacteristic};

    GattService uartService(UARTServiceUUID, charTable, sizeof (charTable) / sizeof (GattCharacteristic *));

    _ble.addService(uartService);

    this->rxCharacteristicHandle = rxCharacteristic.getValueAttribute().getHandle();

    _ble.gattServer().onDataWritten(this, &pfodBleUart::onDataWritten);
    _ble.gattServer().onConfirmationReceived(pfod_on_confirmation);
}

/**
 * clear tx rx buffers
 * called on setup and on disconnect
 */
int pfodBleUart::init() {
    rxBufferHead = 0;
    rxBufferTail = 0;
    txBufferHead = 0;
    txBufferTail = 0;
    pfodTxBufferIdx = 0;
    if ((txBuffer == NULL) || (rxBuffer == NULL)) {
        return MICROBIT_NO_RESOURCES;
    } // else
    return 0; // OK
}


/**
 * bytes available for read
 * @return 0 if none available else >0
 */
size_t pfodBleUart::available(void) {
    return rxBufferedSize(); //(isReadable() ? 1 : 0);
};

/**
 * Read byte from RX buffer
 * @return byte read or -1 if none available
 */
int pfodBleUart::read(void) {
    return getc();
};

/**
 * space available to write without blocking
 * @return number of tx bytes that can be written without blocking
 */
size_t pfodBleUart::txFree(void) {
    return (PFOD_UART_DEFAULT_TX_BUF_SIZE - pfodTxBufferIdx);
};

/**
 * write char, send to ble if have 20 chars OR if char is } or if char is \n
 * @param c
 * @return 1
 */
int pfodBleUart::write(uint8_t c) {
    int rtn = 1;
    if (pfodTxBufferIdx >= PFOD_UART_DEFAULT_TX_BUF_SIZE) {
        return rtn; // should not get this
    }

    pfodTxBuffer[pfodTxBufferIdx++] = c;
    if ((pfodTxBufferIdx >= PFOD_UART_DEFAULT_TX_BUF_SIZE) || (c == ((uint8_t) '}')) || (c == ((uint8_t) '\n'))) {
        int sendRtn = send(pfodTxBuffer, pfodTxBufferIdx); // block here until sent
        UNUSED(sendRtn); // ignore return i.e. if not connected
        pfodTxBufferIdx = 0; // reset idx
    }
    return rtn;
};

/**
 * force send of current bytes
 * @param c
 * @return 1
 */
void pfodBleUart::flush(void) {
    if (pfodTxBufferIdx > 0)  {
        int sendRtn = send(pfodTxBuffer, pfodTxBufferIdx); // block here until sent
        UNUSED(sendRtn); // ignore return i.e. if not connected
        pfodTxBufferIdx = 0; // reset idx
    }
    // else nothing to send
};

/**
 * is this a BLE stream
 * @return true
 */
bool pfodBleUart::isBle(void) {
    return true;
}; 

/**
 * A callback function for whenever a Bluetooth device writes to our RX characteristic.
 */
void pfodBleUart::onDataWritten(const GattWriteCallbackParams *params) {
    if (params->handle == this->rxCharacteristicHandle) {
        uint16_t bytesWritten = params->len;

        for (int byteIterator = 0; byteIterator < bytesWritten; byteIterator++) {
            int newHead = (rxBufferHead + 1) % rxBufferSize;

            if (newHead != rxBufferTail) {
                char c = params->data[byteIterator];

                rxBuffer[rxBufferHead] = c;
                rxBufferHead = newHead;
            }
        }
        MicroBitEvent(PFOD_ID_UART, PFOD_UART_EVT_DATA_AVAILABLE);
    }
}

/**
 * An internal method that copies values from a circular buffer to a linear buffer.
 *
 * @param circularBuff a pointer to the source circular buffer
 * @param circularBuffSize the size of the circular buffer
 * @param linearBuff a pointer to the destination linear buffer
 * @param tailPosition the tail position in the circular buffer you want to copy from
 * @param headPosition the head position in the circular buffer you want to copy to
 *
 * @note this method assumes that the linear buffer has the appropriate amount of
 *       memory to contain the copy operation
 */
void pfodBleUart::circularCopy(uint8_t *circularBuff, uint8_t circularBuffSize, uint8_t *linearBuff, uint16_t tailPosition, uint16_t headPosition) {
    int toBuffIndex = 0;

    while (tailPosition != headPosition) {
        linearBuff[toBuffIndex++] = circularBuff[tailPosition];

        tailPosition = (tailPosition + 1) % circularBuffSize;
    }
}

/**
 * Retreives a single character from our RxBuffer.
 *
 * @param mode the selected mode, one of: ASYNC, SYNC_SPINWAIT, SYNC_SLEEP. Each mode
 *        gives a different behaviour:
 *
 *            ASYNC - Will attempt to read a single character, and return immediately
 *
 *            SYNC_SPINWAIT - will return MICROBIT_INVALID_PARAMETER
 *
 *            SYNC_SLEEP - Will configure the event and block the current fiber until the
 *                         event is received.
 *
 * @return MICROBIT_INVALID_PARAMETER if the mode given is SYNC_SPINWAIT, a character or MICROBIT_NO_DATA
 */
int pfodBleUart::getc() { //mode is alway async here

    if (!isReadable()) {
        return PFOD_UART_NO_DATA;
    }

    char c = rxBuffer[rxBufferTail];

    rxBufferTail = (rxBufferTail + 1) % rxBufferSize;

    return c;
}

/**
 * places a single character into our transmission buffer,
 *
 * @param c the character to transmit
 *
 * @param mode the selected mode, one of: ASYNC, SYNC_SPINWAIT, SYNC_SLEEP. Each mode
 *        gives a different behaviour:
 *
 *            ASYNC - Will copy as many characters as it can into the buffer for transmission,
 *                    and return control to the user.
 *
 *            SYNC_SPINWAIT - will return MICROBIT_INVALID_PARAMETER
 *
 *            SYNC_SLEEP - Will perform a cooperative blocking wait until all
 *                         given characters have been received by the connected
 *                         device.
 *
 * @return the number of characters written, or MICROBIT_NOT_SUPPORTED if there is
 *         no connected device, or the connected device has not enabled indications.
 */
//int pfodMBitUart::putc(char c, MicroBitSerialMode mode) {
//    return (send((uint8_t *) & c, 1, mode) == 1) ? 1 : EOF;
//}

/**
 * Copies characters into the buffer used for Transmitting to the central device.
 *
 * @param buf a buffer containing length number of bytes.
 * @param length the size of the buffer.
 * @param mode the selected mode, one of: ASYNC, SYNC_SPINWAIT, SYNC_SLEEP. Each mode
 *        gives a different behaviour:
 *
 *            ASYNC - Will copy as many characters as it can into the buffer for transmission,
 *                    and return control to the user.
 *
 *            SYNC_SPINWAIT - will return MICROBIT_INVALID_PARAMETER
 *
 *            SYNC_SLEEP - Will perform a cooperative blocking wait until all
 *                         given characters have been received by the connected
 *                         device.
 *
 * @return the number of characters written, or MICROBIT_NOT_SUPPORTED if there is
 *         no connected device, or the connected device has not enabled indications.
 */
// always called with sync_sleep
int pfodBleUart::send(const uint8_t *buf, int length) {
    MicroBitSerialMode mode = SYNC_SLEEP;
    if (length < 1) {
        return 0; // nothing to do
    }

    bool updatesEnabled = false;

    ble.gattServer().areUpdatesEnabled(*txCharacteristic, &updatesEnabled);

    if (!ble.getGapState().connected && !updatesEnabled)
        return MICROBIT_NOT_SUPPORTED;

    int bytesWritten = 0;

    while (bytesWritten < length && ble.getGapState().connected && updatesEnabled) {
        for (int bufferIterator = bytesWritten; bufferIterator < length; bufferIterator++) {
            int nextHead = (txBufferHead + 1) % txBufferSize;

            if (nextHead != txBufferTail) {
                txBuffer[txBufferHead] = buf[bufferIterator];

                txBufferHead = nextHead;

                bytesWritten++;
            }
        }

        int size = txBufferedSize();

        uint8_t temp[size];

        memclr(&temp, size);

        circularCopy(txBuffer, txBufferSize, temp, txBufferTail, txBufferHead);


        if (mode == SYNC_SLEEP)
            fiber_wake_on_event(PFOD_ID_UART, PFOD_UART_DATA_SENT);

        ble.gattServer().write(txCharacteristic->getValueAttribute().getHandle(), temp, size);

        if (mode == SYNC_SLEEP)
            schedule();
        else
            break;

        ble.gattServer().areUpdatesEnabled(*txCharacteristic, &updatesEnabled);
    }

    return bytesWritten;
}

/**
 * Copies characters into the buffer used for Transmitting to the central device.
 *
 * @param s the string to transmit
 * @param mode the selected mode, one of: ASYNC, SYNC_SPINWAIT, SYNC_SLEEP. Each mode
 *        gives a different behaviour:
 *
 *            ASYNC - Will copy as many characters as it can into the buffer for transmission,
 *                    and return control to the user.
 *
 *            SYNC_SPINWAIT - will return MICROBIT_INVALID_PARAMETER
 *
 *            SYNC_SLEEP - Will perform a cooperative blocking wait until all
 *                         given characters have been received by the connected
 *                         device.
 *
 * @return the number of characters written, or MICROBIT_NOT_SUPPORTED if there is
 *         no connected device, or the connected device has not enabled indications.
 */
//int pfodMBitUart::send(ManagedString s, MicroBitSerialMode mode) {
//    return send((uint8_t *) s.toCharArray(), s.length(), mode);
//}

/**
 * Determines if we have space in our rxBuff.
 *
 * @return 1 if we have data available to read, 0 if we do not.
 *
 * @note the reason we do not wrap the super's readable() method is so that we
 *       don't interfere with communities that use manual calls to uBit.serial.readable()
 */
int pfodBleUart::isReadable() {
    return (rxBufferTail != rxBufferHead) ? 1 : 0;
}

/**
 * @return The currently buffered number of bytes in our rxBuff.
 */
int pfodBleUart::rxBufferedSize() {
    if (rxBufferTail > rxBufferHead)
        return (rxBufferSize - rxBufferTail) + rxBufferHead;

    return rxBufferHead - rxBufferTail;
}

/**
 * @return The currently buffered number of bytes in our txBuff.
 */
int pfodBleUart::txBufferedSize() {
    if (txBufferTail > txBufferHead)
        return (txBufferSize - txBufferTail) + txBufferHead;

    return txBufferHead - txBufferTail;
}
