/*
 pfodParserMicroBit library V1.0   20th October 2016 (www.pfod.com.au)
 (c)2016 Forward Computing and Control Pty. Ltd.
 This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 This code may be freely used for both private and commercial use.
 Provide this copyright is maintained.
 */

/*
This code derived from BBC micro:bit C++ MicroBitUARTService.h used under the following licence
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
 */

#ifndef PFOD_MBIT_UART_H
#define PFOD_MBIT_UART_H

/**
 * This header file contains IDs and event codes use for Bluetooth communication.
 */


#include "mbed.h"
#include "ble/UUID.h"
#include "ble/BLE.h"
#include "MicroBitConfig.h"
#include "MicroBitSerial.h"
#include "pfodParserStream.h"

/**
 * Class definition for the custom MicroBit UART Service.
 * Provides a BLE service that acts as a UART port, enabling the reception and transmission
 * of an arbitrary number of bytes.
 */
class pfodBleUart : public pfodParserStream {
public:
    int init(); // clears tx/rx buffer
    size_t available(void); // return bytes available for read()
    int read(void); // read an input byte from rx buffer. Note: spins if none available
    size_t txFree(void); // return space available for write();
    int write(uint8_t c); // write to tx buffer. Note: spins if no space to write
    void flush(void); // force write of bytes
    bool isBle(void); // is this a BLE stream

    /**
     * Constructor for the UARTService.
     * @param _ble an instance of BLEDevice
     * @param rxBufferSize the size of the rxBuffer
     *
     * @note The default TX size is PFOD_UART_DEFAULT_BUF_SIZE (20 bytes).
     */
    pfodBleUart(BLEDevice &_ble, int rxBufSize);


    /**
     * Retreives a single character from our RxBuffer.
     *
     * @param mode the selected mode, one of: ASYNC, SYNC_SPINWAIT, SYNC_SLEEP. Each mode
     *        gives a different behaviour:
     *
     *            ASYNC - Will attempt to read a single character, and return immediately
     *
     *            SYNC_SPINWAIT - will return MICROBIT_INVALID_PARAMETER
     *
     *            SYNC_SLEEP - Will configure the event and block the current fiber until the
     *                         event is received.
     *
     * @return MICROBIT_INVALID_PARAMETER if the mode given is SYNC_SPINWAIT, a character or MICROBIT_NO_DATA
     */
    int getc();

private:
    uint8_t* rxBuffer;
    uint8_t* txBuffer;
    uint8_t rxBufferHead;
    uint8_t rxBufferTail;
    size_t rxBufferSize;
    uint8_t txBufferSize;
    uint32_t rxCharacteristicHandle;
    // Bluetooth stack we're running on.
    BLEDevice &ble;
    // write limits sends to 20 bytes
    static const size_t PFOD_UART_DEFAULT_TX_BUF_SIZE = 20;
    size_t pfodTxBufferIdx;
    uint8_t pfodTxBuffer[PFOD_UART_DEFAULT_TX_BUF_SIZE]; // array for sending chars 20 at a time

    /**
     * A callback function for whenever a Bluetooth device writes to our TX characteristic.
     */
    void onDataWritten(const GattWriteCallbackParams *params);

    /**
     * An internal method that copies values from a circular buffer to a linear buffer.
     *
     * @param circularBuff a pointer to the source circular buffer
     * @param circularBuffSize the size of the circular buffer
     * @param linearBuff a pointer to the destination linear buffer
     * @param tailPosition the tail position in the circular buffer you want to copy from
     * @param headPosition the head position in the circular buffer you want to copy to
     *
     * @note this method assumes that the linear buffer has the appropriate amount of
     *       memory to contain the copy operation
     */
    void circularCopy(uint8_t *circularBuff, uint8_t circularBuffSize, uint8_t *linearBuff, uint16_t tailPosition, uint16_t headPosition);
    /**
     * Determines if we have space in our rxBuff.
     *
     * @return 1 if we have space, 0 if we do not.
     */
    int isReadable();

    /**
     * @return The currently buffered number of bytes in our rxBuff.
     */
    int rxBufferedSize();

    /**
     * @return The currently buffered number of bytes in our txBuff.
     */
    int txBufferedSize();
    /**
     * Copies characters into the buffer used for Transmitting to the central device.
     *
     * @param buf a buffer containing length number of bytes.
     * @param length the size of the buffer.
     *
     * @return the number of characters written, or MICROBIT_NOT_SUPPORTED if there is
     *         no connected device, or the connected device has not enabled indications.
     */
    int send(const uint8_t *buf, int length);

};

extern const uint8_t UARTServiceBaseUUID[UUID::LENGTH_OF_LONG_UUID];
extern const uint16_t UARTServiceShortUUID;
extern const uint16_t UARTServiceTXCharacteristicShortUUID;
extern const uint16_t UARTServiceRXCharacteristicShortUUID;

extern const uint8_t UARTServiceUUID[UUID::LENGTH_OF_LONG_UUID];
extern const uint8_t UARTServiceUUID_reversed[UUID::LENGTH_OF_LONG_UUID];

extern const uint8_t UARTServiceTXCharacteristicUUID[UUID::LENGTH_OF_LONG_UUID];
extern const uint8_t UARTServiceRXCharacteristicUUID[UUID::LENGTH_OF_LONG_UUID];

#endif
