/*
 pfodParserMicroBit library V1.0   20th October 2016 (www.pfod.com.au)
 (c)2016 Forward Computing and Control Pty. Ltd.
 This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 This code may be freely used for both private and commercial use.
 Provide this copyright is maintained.
 */

/**
pfodParser for Arduino
The pfodParser parses messages of the form
 { cmd ` arg1 ` arg2 ` ... }
 or
 { cmd ~ arg1}
 or
 { cmd }
 The args are optional
 This is a complete paser for ALL commands a pfodApp will send to a pfodDevice
 see www.pfod.com.au  for more details.

  pfodParser adds about 482 uint8_ts to the program and uses about 260 uint8_ts RAM

The pfodParser parses messages of the form
 { cmd ` arg1 ` arg2 ` ... }
 or
 { cmd ~ arg1}
 or
 { cmd }
The message is parsed into the args array by replacing '|', '`' and '}' with '/0' (null)
When the the end of message } is seen
  parse() returns the first uint8_t of the cmd
  getCmd() returns a pointer to the null terminated cmd
  skipCmd() returns a pointer to the first arg (null terminated)
      or a pointer to null if there are no args
  getArgsCount() returns the number of args found.
These calls are valid until the start of the next msg { is parsed.
At which time they are reset to empty command and no args.

 */

/* the print fns shown here are derived from Arduino Print.cpp
 * Modifications by Matthew Ford (c)2016 Forward Computing and Control Pty.Ltd.
 * 
  Print.h - Base class that provides print() and println()
  Copyright (c) 2008 David A. Mellis.  All right reserved.

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <string.h>
#include "pfodParser.h"

#define UNUSED(x) (void)(x)

void pfodParser::waitForParserStreamLock() {
    if (isParserStreamLocked()) {
        do {
            fiber_wait_for_event(PFOD_ID_UART, PFOD_PARSER_STREAM_FREE);
        } while (isParserStreamLocked());
    }
    lockParserStream();
}

/**
 * Locks the mutex so that others can't use this serial instance for transmission
 */
void pfodParser::lockParserStream() {
    parserStreamLocked = true;
}

/**
 * Unlocks the mutex so that others can use this serial instance for reception
 */
void pfodParser::unlockParserStream() {
    parserStreamLocked = false;
    MicroBitEvent(PFOD_ID_UART, PFOD_PARSER_STREAM_FREE); // wake next fiber, in order, that is waiting to use Serial
    uBitPtr->sleep(0); // put this fiber on the end of the queue so other waiting fibers get a go at the resource
}

bool pfodParser::isParserStreamLocked() {
    return parserStreamLocked;
}

void pfodParser::processCmd() {
    // nothing here pfodMenu overrides this with real processCmd
    uint8_t cmd = *getCmd();
    if (cmd != 0) { // have parsed a complete msg { to }
        print("{}");
    }
}

void pfodParser::parseCmd() {
    uint8_t cmd = parse(); // parse incoming data from connection
    // parser returns non-zero when a pfod command is fully parsed
    if (cmd == 0) return;
    // else see if we can take lock to send back msg
    waitForParserStreamLock();
    processCmd();
    unlockParserStream();
}

int pfodParser::swap01(int in) {
    return (in == 0) ? 1 : 0;
}


pfodParser::pfodParser() {
}

void pfodParser::init(MicroBit* _uBitPtr, const char *_version, int maxRxMsgSize ) {
    uBitPtr = _uBitPtr;
    if (maxRxMsgSize > 255) {
         maxRxMsgSize = 255;
    }
    pfodMaxMsgLen = maxRxMsgSize;
    args = new uint8_t[pfodMaxMsgLen+1];
    setVersion(_version);
    ioStream = NULL;
    emptyVersion[0] = 0;
    initCmdParser();
    parserStreamLocked = false;
}

void pfodParser::setStream(pfodParserStream* _stream) {
    ioStream = _stream;
    parserStreamLocked = false;
    connected = false;
}

/**
 * Note: this must NOT null the iostream
 */
void pfodParser::initCmdParser() {
    //connected = false;
    argsCount = 0;
    argsIdx = 0;
    args[0] = 0; // no cmd yet
    args[1] = 0; //
    args[pfodMaxMsgLen] = 0; // double terminate
    args[pfodMaxMsgLen - 1] = 0;
    cmdStart = args; // if no version
    versionStart = emptyVersion; // not used if : not found
    parserState = pfodWaitingForStart; // not started yet pfodInMsg when have seen {
    refresh = false;
}

void pfodParser::connect(void) { //pfodParserStream* ioPtr) {
    connected = true;
}
bool pfodParser::isConnected() {
    return connected;
}

void pfodParser::closeConnection() {
    ioStream->init(); // clear all tx buffers, leave rx unchanged.
    initCmdParser(); // connected == false
    connected = false;
}

pfodParserStream* pfodParser::getPfodAppStream() {
    return ioStream;
}

size_t pfodParser::write(uint8_t c) {
    if ((!connected) || (!ioStream)) {
        return 1; // cannot write if ioStream null but just pretend to
    }
    return ioStream->write(c);
}

// print support

int pfodParser::println(void) {
    // skips print if not connected
    int rtn = write('\r');
    rtn += write('\n');
    return rtn;
}

int pfodParser::print(char c) {
    // skips print if not connected
    return write((uint8_t) c);
}

int pfodParser::print(double number, uint8_t digits) {
    // skips print if not connected
  int rtn = 0;
  
  if (isnan(number)) return print("nan");
  if (isinf(number)) return print("inf");
  if (number > 4294967040.0) return print("ovf");  // constant determined empirically
  if (number <-4294967040.0) return print("ovf");  // constant determined empirically
  
  // Handle negative numbers
  if (number < 0.0)  {
     rtn += write('-');
     number = -number;
  }

  // Round correctly so that print(1.999, 2) prints as "2.00"
  double rounding = 0.5;
  for (uint8_t i=0; i<digits; ++i) {
    rounding /= 10.0;
  }
  number += rounding;

  // Extract the integer part of the number and print it
  const unsigned long int_part = (unsigned long)number;
  double remainder = number - (double)int_part;
  rtn += print(int_part);

  // Print the decimal point, but only if there are digits beyond
  if (digits > 0) {
    rtn += write('.'); 
  }

  // Extract digits from the remainder one at a time
  while (digits-- > 0) {
    remainder *= 10.0;
    int toPrint = int(remainder);
    rtn += write((char)('0'+toPrint));
    remainder -= toPrint; 
  } 
  return rtn;
}

int pfodParser::print(const char *str) {
    // skips print if not connected
    int rtn = 0;
    while (*str) {
        rtn += write((uint8_t) * str++);
    }
    return rtn;
}

int pfodParser::print(ManagedString s) {
    uint8_t *str = (uint8_t *) s.toCharArray();
    str[s.length()] = 0; // make sure it is terminated;
    return print((const char*) str);
}

int pfodParser::print(const int i) {
    return print((long) i);
};

int pfodParser::print(const unsigned long lin) {// skips print if not connected
    // skips print if not connected
    char buf[3 * sizeof (unsigned long) + 2]; // Assumes max 3char per byte plus sign plus null terminator byte.  
    char *str = &buf[sizeof (buf) - 1];
    unsigned long l = lin;
    *str = '\0';
    do {
        char c = l % 10;
        l /= 10;
        *--str = c < 10 ? c + '0' : c + 'A' - 10;
    } while (l);
    return print(str);
}

int pfodParser::print(const long lin) {
    // skips print if not connected
    char buf[3 * sizeof (long) + 2]; // Assumes max 3char per byte plus sign plus null terminator byte.  
    char *str = &buf[sizeof (buf) - 1];
    long l = lin;
    *str = '\0';
    uint8_t neg = 0;
    if (l < 0) {
        neg = 1;
        l = -l;
    }
    do {
        char c = l % 10;
        l /= 10;
        *--str = c < 10 ? c + '0' : c + 'A' - 10;
    } while (l);
    if (neg) {
        *--str = '-';
    }
    return print(str);

}

void pfodParser::flush() {
    if ((!connected) || (!ioStream)) {
        return; // cannot write if ioStream null but just pretend to
    }
    ioStream->flush(); // force write of buffered bytes if any
}

float pfodParser::getPlotVarScaling(long varMax, long varMin, float displayMax, float displayMin) {
    long varRange = varMax-varMin;
    if (varRange == 0) { varRange = 1; } // prevent divide by zero
    return (displayMax-displayMin)/((float)varRange);
}


void pfodParser::setCmd(uint8_t cmd) {
    initCmdParser();
    args[0] = cmd;
    args[1] = 0;
    cmdStart = args;
    versionStart = emptyVersion; // leave refresh unchanged
}

/**
 * Return pointer to start of message return start of cmd
 */
uint8_t* pfodParser::getCmd() {
    return cmdStart;
}

/**
 * msg starts with {: the : is dropped from the cmd
 */
bool pfodParser::isRefresh() {
    return refresh;
}

const char* pfodParser::getVersion() {
    return version;
}

void pfodParser::setVersion(const char* _version) {
    version = _version;
}

void pfodParser::sendVersion() {
    print('~');
    print(getVersion());
}

/**
 * Return pointer to first arg (or pointer to null if no args)
 *
 * Start at args[0] and scan for first null
 * if argsCount > 0 increment to point to  start of first arg
 * else if argsCount == 0 leave pointing to null
 */
uint8_t* pfodParser::getFirstArg() {
    uint8_t* idxPtr = cmdStart;
    while (*idxPtr != 0) {
        ++idxPtr;
    }
    if (argsCount > 0) {
        ++idxPtr;
    }
    return idxPtr;
}

/**
 * Return pointer to next arg or pointer to null if end of args
 * Need to call getFirstArg() first
 * Start at current pointer and scan for first null
 * if scanned over a non empty arg then skip terminating null and return
 * pointer to next arg, else return start if start points to null already
 */
uint8_t* pfodParser::getNextArg(uint8_t *start) {
    uint8_t* idxPtr = start;
    while (*idxPtr != 0) {
        ++idxPtr;
    }
    if (idxPtr != start) {
        ++idxPtr; // skip null
    } // else this was the last arg
    return idxPtr;
}

/**
 * Return number of args in last parsed msg
 */
uint8_t pfodParser::getArgsCount() {
    return argsCount;
}

/**
 * pfodWaitingForStart if outside msg
 * pfodMsgStarted if just seen opening {
 * pfodInMsg in msg after {
 * pfodMsgEnd if just seen closing }
 */
uint8_t pfodParser::getParserState() {
    if ((parserState == pfodWaitingForStart) ||
            (parserState == pfodMsgStarted) ||
            (parserState == pfodInMsg) ||
            (parserState == pfodMsgEnd)) {
        return parserState;
    } //else {
    return pfodInMsg;
}

uint8_t pfodParser::parse() {
    uint8_t rtn = 0;
    if (!ioStream) {
        return rtn;
    }
    while (ioStream->available()) {
        int in = ioStream->read();
        rtn = parse((uint8_t) in);
        if (rtn != 0) {
            // found msg
            return rtn;
        }
    }
    return rtn;
}

/**
 * parse
 * Inputs:
 * uint8_t in -- uint8_t read from Serial port
 * Return:
 * return 0 if complete message not found yet
 * else return first uint8_t of cmd when see closing }
 * or ignore msg if pfodMaxCmdLen uint8_ts after {
 * On non-zero return args[] contains
 * the cmd null terminated followed by the args null terminated
 * argsCount is the number of args
 *
 * parses
 * { cmd | arg1 ` arg2 ... }
 * { cmd ` arg1 ` arg2 ... }
 * { cmd ~ arg1 ~ arg2 ... }
 * save the cmd in args[] replace |, ~ and ` with null (\0)
 * then save arg1,arg2 etc in args[]
 * count of args saved in argCount
 * on seeing } return first uint8_t of cmd
 * if no } seen for pfodMaxCmdLen uint8_ts after starting { then
 * ignore msg and start looking for { again
 *
 * States:
 * before {   parserState == pfodWaitingForStart
 * when   { seen parserState == pfodInMsg
 */
uint8_t pfodParser::parse(uint8_t in) {
    if (in == 0xff) {
        // note 0xFF is not a valid utf-8 uint8_t
        // but is returned by underlying stream if start or end of connection
        // NOTE: Stream.read() is wrapped in while(Serial.available()) so should not get this
        // unless explicitlly added to stream buffer
        initCmdParser(); // clean out last partial msg if any
        return 0;
    }
    if ((parserState == pfodWaitingForStart) || (parserState == pfodMsgEnd)) {
        parserState = pfodWaitingForStart;
        if (in == pfodMsgStarted) { // found {
            initCmdParser(); // clean out last cmd
            parserState = pfodMsgStarted;
        }
        // else ignore this uint8_t as waiting for start {
        // always reset counter if waiting for {
        return 0;
    }

    // else have seen {
    if ((argsIdx >= (pfodMaxMsgLen - 2)) && // -2 since { never stored
            (in != pfodMsgEnd)) {
        // ignore this msg and reset
        // should not happen as pfodApp should limit
        // msgs sent to pfodDevice to <=255 uint8_ts
        initCmdParser();
        return 0;
    }
    // first uint8_t after opening {
    if (parserState == pfodMsgStarted) {
        parserState = pfodInMsg;
        if (in == pfodRefresh) {
            refresh = true;
            return 0; // skip this uint8_t if get {:
        }
    }
    // else continue. Check for version:
    if ((in == pfodRefresh) && (versionStart != args)) {
        // found first : set version pointer
        args[argsIdx++] = 0;
        versionStart = args;
        cmdStart = args + argsIdx; // next uint8_t after :
        refresh = (strcmp((const char*) versionStart, version) == 0);
        return 0;
    }

    // else process this msg uint8_t
    // look for special uint8_ts | ' } 
    if ((in == pfodMsgEnd) || (in == pfodBar) || (in == pfodTilda) || (in == pfodAccent)) {
        args[argsIdx++] = 0;
        if (parserState == pfodArgStarted) {
            // increase count if was parsing arg
            argsCount++;
        }
        if (in == pfodMsgEnd) {
            parserState = pfodMsgEnd; // reset state
            args[argsIdx++] = 0;
            // return command uint8_t found
            // this will return 0 when parsing {} msg
            return cmdStart[0];
        } else {
            parserState = pfodArgStarted;
        }
        return 0;
    }
    // else normal uint8_t
    args[argsIdx++] = in;
    return 0;
}

/**
 * parseLong
 * will parse between  -2,147,483,648 to 2,147,483,647
 * No error checking done.
 * will return 0 for empty string, i.e. first uint8_t == null
 *
 * Inputs:
 *  idxPtr - uint8_t* pointer to start of uint8_ts to parse
 *  result - long* pointer to where to store result
 * return
 *   uint8_t* updated pointer to uint8_ts after skipping terminator
 *
 */
uint8_t* pfodParser::parseLong(uint8_t* idxPtr, long *result) {
    long rtn = 0;
    bool neg = false;
    while (*idxPtr != 0) {
        if (*idxPtr == '-') {
            neg = true;
        } else {
            rtn = (rtn << 3) + (rtn << 1); // *10 = *8+*2
            rtn = rtn + (*idxPtr - '0');
        }
        ++idxPtr;
    }
    if (neg) {
        rtn = -rtn;
    }
    *result = rtn;
    return ++idxPtr; // skip null
}


//void pfodParser::setDebugStream(Print* debugOut) {
//  ; // does nothing
//}

