/*
 pfodParserMicroBit library V1.0   20th October 2016 (www.pfod.com.au)
 (c)2016 Forward Computing and Control Pty. Ltd.
 This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 This code may be freely used for both private and commercial use.
 Provide this copyright is maintained.
 */

#ifndef pfodParser_h
#define pfodParser_h

/**
pfodParser for Arduino
 Parses commands of the form { cmd | arg1 ` arg2 ... }
 Arguments are separated by `
 The | and the args are optional
 This is a complete paser for ALL commands a pfodApp will send to a pfodDevice
 see www.pfod.com.au  for more details.

  pfodParser adds about 482 uint8_ts to the program and uses about 260 uint8_ts RAM

The pfodParser parses messages of the form
 { cmd | arg1 ` arg2 ` ... }
The message is parsed into the args array by replacing '|', '`' and '}' with '/0' (null)
When the the end of message } is seen
  parse() returns the first uint8_t of the cmd
  getCmd() returns a pointer to the null terminated cmd
  skipCmd() returns a pointer to the first arg (null terminated)
      or a pointer to null if there are no args
  getArgsCount() returns the number of args found.
These calls are valid until the start of the next msg { is parsed.
At which time they are reset to empty command and no args.

 */
#include "pfodParserStream.h"
#include "pfodParserUtils.h"
#include "ManagedString.h"
#include "MicroBit.h"

class pfodParser {
protected:
    pfodParser(void); 
    int swap01(int in);
    void parseCmd(void); // called by MicroBitEvent(PFOD_ID_UART, PFOD_UART_EVT_DATA_AVAILABLE)
    virtual void processCmd(void);

    // print support
    int println(void); // skips print if not connected
    int print(char c); // skips print if not connected
    int print(const char *str); // skips print if not connected
    int print(const long l); // skips print if not connected
    int print(const int i); // skips print if not connected
    int print(const unsigned long l); // skips print if not connected
    int print(double f, uint8_t digits = 2); // skips print if not connected
    int print(ManagedString str);
    // bool isConnected(); // true if uart connected
    virtual void connect(void);
    virtual void closeConnection(void);
    bool isConnected(void);
    uint8_t parse(void);
    bool isRefresh(void); // starts with {version: and the version matches this parser's version
    const char* getVersion(void);
    void setVersion(const char* version); // no usually needed
    void sendVersion(void); // send ~ version to parser.print
    uint8_t* getCmd();
    uint8_t* getFirstArg(void);
    uint8_t* getNextArg(uint8_t *start);
    uint8_t getArgsCount(void);
    uint8_t* parseLong(uint8_t* idxPtr, long *result);
    /**
     * pfodWaitingForStart if outside msg
     * pfodMsgStarted if just seen opening {
     * pfodInMsg in msg after {
     * pfodMsgEnd if just seen closing }
     */
    void setCmd(uint8_t cmd);
    static const uint8_t pfodWaitingForStart = 0xff;
    static const uint8_t pfodMsgStarted = '{';
    static const uint8_t pfodRefresh = ':';
    static const uint8_t pfodInMsg = 0;
    static const uint8_t pfodMsgEnd = '}';
    //    void setDebugStream(Print* debugOut); // does nothing
    size_t write(uint8_t c);
    void flush(void);
    float getPlotVarScaling(long varMax, long varMin, float displayMax, float displayMin);
    pfodParserStream* getPfodAppStream(); // get the command response stream we are writing to

    // this is returned if pfodDevice should drop the connection
    // only returned by pfodParser in read() returns -1
    void init(MicroBit* _uBitPtr, const char *_version, int maxRxMsgSize );
    void initCmdParser(void);
    uint8_t parse(uint8_t in);
    void setStream(pfodParserStream* _stream);
    MicroBit* uBitPtr;
    void waitForParserStreamLock();
    void unlockParserStream();
    uint8_t pfodMaxMsgLen;// = 0xff; // == 255, if no closing } by now ignore msg

private:
    bool connected; // true if connect called or have seen {.} false if closeConnection() called or have seen {!)
    uint8_t getParserState(void);
    bool isParserStreamLocked(void);
    void lockParserStream(void);
    bool parserStreamLocked; // lock for data and parser
    //static const uint8_t DisconnectNow = '!';
    pfodParserStream* ioStream;
    // you can reduce the value if you are handling smaller messages
    // but never increase it.
    uint8_t emptyVersion[1];
    uint8_t argsCount; // no of arguments found in msg
    uint8_t argsIdx;
    uint8_t parserState;
    uint8_t *args;//[pfodMaxMsgLen + 1]; // allow for trailing null
    uint8_t *versionStart;
    uint8_t *cmdStart;
    bool refresh;
    const char *version;
    static const uint8_t pfodBar = (uint8_t) '|';
    static const uint8_t pfodTilda = (uint8_t) '~';
    static const uint8_t pfodAccent = (uint8_t) '`';
    static const uint8_t pfodArgStarted = 0xfe;
};

#endif // pfodParser_h

