/*
 pfodParserMicroBit library V1.0   20th October 2016 (www.pfod.com.au)
 (c)2016 Forward Computing and Control Pty. Ltd.
 This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 This code may be freely used for both private and commercial use.
 Provide this copyright is maintained.
 */

/*
This code derived from BBC micro:bit C++ MicroBitUARTService.cpp and others and is used under the following licence
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
 */

/**
 * Class definition for the custom MicroBit UART Service.
 * Provides a BLE service that acts as a UART port, enabling the reception and transmission
 * of an arbitrary number of bytes.
 */

#include "pfodSerialUart.h"
#include "mbed.h"
#include "MicroBitSerial.h"
#include "ErrorNo.h"
#include "MicroBitComponent.h"
#include "MicroBitFiber.h"
#include "NotifyEvents.h"

//#define UNUSED(x) (void)(x)

uint8_t pfodSerialUart::status = 0;

int pfodSerialUart::baudrate = 0;

/**
 * Constructor.
 * Create an instance of MicroBitSerial
 *
 * @param tx the Pin to be used for transmission
 *
 * @param rx the Pin to be used for receiving data
 *
 * @param rxBufferSize the size of the buffer to be used for receiving bytes
 *
 * @param txBufferSize the size of the buffer to be used for transmitting bytes
 *
 * @code
 * MicroBitSerial serial(USBTX, USBRX);
 * @endcode
 * @note the default baud rate is 115200. More API details can be found:
 *       -https://github.com/mbedmicro/mbed/blob/master/libraries/mbed/api/SerialBase.h
 *       -https://github.com/mbedmicro/mbed/blob/master/libraries/mbed/api/RawSerial.h
 *
 *       Buffers aren't allocated until the first send or receive respectively.
 */
pfodSerialUart::pfodSerialUart(PinName tx, PinName rx, uint8_t txBufferSize) : RawSerial(tx, rx) {
    int rxBufferSize = 256;
    // + 1 so there is a usable buffer size, of the size the user requested.
    this->rxBuffSize = rxBufferSize + 1;
    this->txBuffSize = txBufferSize + 1;
    this->rxBuff = NULL;
    this->txBuff = NULL;

    setRxBufferSize(this->rxBuffSize);
    setTxBufferSize(this->txBuffSize);


    this->rxBuffHead = 0;
    this->rxBuffTail = 0;

    this->txBuffHead = 0;
    this->txBuffTail = 0;

    this->rxBuffHeadMatch = -1;

    this->baud(MICROBIT_SERIAL_DEFAULT_BAUD_RATE);

#if CONFIG_ENABLED(MICROBIT_DBG)
    SERIAL_DEBUG = this;
#endif

}

/** 
 * initialize and clears buffers and interrupts
 * called on setup and on disconnect
 * returns MICROBIT_NO_RESOURCES if malloc fails
 */
int pfodSerialUart::init() {
    int rtn = MICROBIT_OK;
    rtn = initialiseRx();
    rtn |= initialiseTx();
    return rtn;
}

/**
 * The number of bytes currently stored in our rx buffer waiting to be digested,
 * by the user.
 *
 * @return The currently buffered number of bytes in our rxBuff.
 */
size_t pfodSerialUart::available() {
    if (rxBuffTail > rxBuffHead)
        return (rxBuffSize - rxBuffTail) + rxBuffHead;

    return (rxBuffHead - rxBuffTail);
}

/**
 * Reads a single character from the rxBuff
 *
 * mode is ALWAYS ASYNC - A character is read from the rxBuff if available, if there
 *                    are no characters to be read OR if another thread is using this serial
 * , a value of PFOD_UART_NO_DATA is returned immediately.
 *
 *  always guard call with available()>0
 *
 * @return a character, PFOD_UART_NO_DATA if another fiber is using the serial instance for reception,
 *         or if  the rx buffer is empty and the mode given is ASYNC.
 */
int pfodSerialUart::read() {
    if (rxInUse()) {
        return PFOD_UART_NO_DATA;
    }
    lockRx();
    int c = getChar();
    unlockRx();
    return c;
}

/**
 * The number of bytes currently stored in our tx buffer waiting to be transmitted
 * by the hardware.
 *
 * @return The currently buffered number of bytes in our txBuff.
 */
size_t pfodSerialUart::txFree() {
    if (txBuffTail > txBuffHead)
        return (txBuffSize - ((txBuffSize - txBuffTail) + txBuffHead));

    return (txBuffSize - (txBuffHead - txBuffTail));
}

/**
 * Sends a single character over the serial line.
 *
 * @param c the character to send
 *
 *     the character is copied into the txBuff and the fiber sleeps
 *                         until the character has been sent. This allows other fibers
 *                         to continue execution.
 *
 * @return the number of bytes written, or PFOD_UART_NO_DATA if another fiber
 *         is using the serial instance for transmission.
 */
int pfodSerialUart::write(uint8_t c) {
    if (txInUse())
        return PFOD_UART_NO_DATA;

    lockTx();

    uint8_t toTransmit[2] = {c, '\0'};

    int bytesWritten = setTxInterrupt(toTransmit, 1);

    send();

    unlockTx();

    return bytesWritten;
}

void pfodSerialUart::flush(void) {
    // nothing to do here all writes are SPIN_WAIT
}

/**
 * is this a BLE stream
 * @return false
 */
bool pfodSerialUart::isBle(void) {
    return false;
}; 

/**
 * An internal interrupt callback for MicroBitSerial configured for when a
 * character is received.
 *
 * Each time a character is received fill our circular buffer!
 */
void pfodSerialUart::dataReceived() {
    if (!(status & MICROBIT_SERIAL_RX_BUFF_INIT)) {
        return;
    }
    bool sendEvent = false;
    if (rxBuffTail == rxBuffHead) {
        // was empty
        sendEvent = true;
    }
    char c = getc();
    uint16_t newHead = (rxBuffHead + 1) % rxBuffSize;

    //look ahead to our newHead value to see if we are about to collide with the tail
    if (newHead != rxBuffTail) {
        //if we are not, store the character, and update our actual head.
        this->rxBuff[rxBuffHead] = c;
        rxBuffHead = newHead;
    }
    if (sendEvent) {
        MicroBitEvent(PFOD_ID_UART, PFOD_UART_EVT_DATA_AVAILABLE);
    }
}

/**
 * An internal interrupt callback for MicroBitSerial.
 *
 * Each time the Serial module's buffer is empty, write a character if we have
 * characters to write.
 */
void pfodSerialUart::dataWritten() {
    if (txBuffTail == txBuffHead || !(status & MICROBIT_SERIAL_TX_BUFF_INIT))
        return;

    //send our current char
    putc(txBuff[txBuffTail]);

    uint16_t nextTail = (txBuffTail + 1) % txBuffSize;

    //unblock any waiting fibers that are waiting for transmission to finish.
    if (nextTail == txBuffHead) {
        MicroBitEvent(PFOD_ID_UART, PFOD_UART_DATA_SENT);
        detach(Serial::TxIrq);
    }

    //update our tail!
    txBuffTail = nextTail;
}

/**
 * An internal method to configure an interrupt on tx buffer and also
 * a best effort copy operation to move bytes from a user buffer to our txBuff
 *
 * @param string a pointer to the first character of the users' buffer.
 *
 * @param len the length of the string, and ultimately the maximum number of bytes
 *        that will be copied dependent on the state of txBuff
 *
 * @param mode determines whether to configure the current fiber context or not. If
 *             The mode is SYNC_SPINWAIT, the context will not be configured, otherwise
 *             no context will be configured.
 *
 * @return the number of bytes copied into the buffer.
 */
int pfodSerialUart::setTxInterrupt(uint8_t *string, int len) {
    MicroBitSerialMode mode = SYNC_SLEEP;
    int copiedBytes = 0;

    for (copiedBytes = 0; copiedBytes < len; copiedBytes++) {
        uint16_t nextHead = (txBuffHead + 1) % txBuffSize;
        if (nextHead != txBuffTail) {
            this->txBuff[txBuffHead] = string[copiedBytes];
            txBuffHead = nextHead;
        } else
            break;
    }

    if (mode != SYNC_SPINWAIT)
        fiber_wake_on_event(PFOD_ID_UART, PFOD_UART_DATA_SENT);

    //set the TX interrupt
    attach((pfodSerialUart*)this, &pfodSerialUart::dataWritten, Serial::TxIrq);

    return copiedBytes;
}

/**
 * Locks the mutex so that others can't use this serial instance for reception
 */
void pfodSerialUart::lockRx() {
    status |= MICROBIT_SERIAL_RX_IN_USE;
}

/**
 * Locks the mutex so that others can't use this serial instance for transmission
 */
void pfodSerialUart::lockTx() {
    status |= MICROBIT_SERIAL_TX_IN_USE;
}

/**
 * Unlocks the mutex so that others can use this serial instance for reception
 */
void pfodSerialUart::unlockRx() {
    status &= ~MICROBIT_SERIAL_RX_IN_USE;
}

/**
 * Unlocks the mutex so that others can use this serial instance for transmission
 */
void pfodSerialUart::unlockTx() {
    status &= ~MICROBIT_SERIAL_TX_IN_USE;
}

/**
 * We do not want to always have our buffers initialised, especially if users to not
 * use them. We only bring them up on demand.
 */
int pfodSerialUart::initialiseRx() {
    if ((status & MICROBIT_SERIAL_RX_BUFF_INIT)) {
        //ensure that we receive no interrupts after freeing our buffer
        detach(Serial::RxIrq);
        free(this->rxBuff);
    }

    status &= ~MICROBIT_SERIAL_RX_BUFF_INIT;

    if ((this->rxBuff = (uint8_t *) malloc(rxBuffSize)) == NULL)
        return MICROBIT_NO_RESOURCES;

    this->rxBuffHead = 0;
    this->rxBuffTail = 0;

    //set the receive interrupt
    status |= MICROBIT_SERIAL_RX_BUFF_INIT;
    attach((pfodSerialUart*)this, &pfodSerialUart::dataReceived, Serial::RxIrq);

    return MICROBIT_OK;
}

/**
 * We do not want to always have our buffers initialised, especially if users to not
 * use them. We only bring them up on demand.
 */
int pfodSerialUart::initialiseTx() {
    if ((status & MICROBIT_SERIAL_TX_BUFF_INIT)) {
        //ensure that we receive no interrupts after freeing our buffer
        detach(Serial::TxIrq);
        free(this->txBuff);
    }

    status &= ~MICROBIT_SERIAL_TX_BUFF_INIT;

    if ((this->txBuff = (uint8_t *) malloc(txBuffSize)) == NULL)
        return MICROBIT_NO_RESOURCES;

    this->txBuffHead = 0;
    this->txBuffTail = 0;

    status |= MICROBIT_SERIAL_TX_BUFF_INIT;

    return MICROBIT_OK;
}

/**
 * An internal method that puts the fiber to sleep 
 *
 */
void pfodSerialUart::send() {
    fiber_sleep(0);
}

/**
 * Reads a single character from the rxBuff
 * if there are no characters to be read, a value of PFOD_UART_NO_DATA is returned immediately.
 *
 * @return a character from the circular buffer, or PFOD_UART_NO_DATA is there
 *         are no characters in the buffer.
 */
int pfodSerialUart::getChar(void) {
    MicroBitSerialMode mode = ASYNC;
    if (mode == ASYNC) {
        if (!isReadable())
            return PFOD_UART_NO_DATA;
    }

    char c = rxBuff[rxBuffTail];

    rxBuffTail = (rxBuffTail + 1) % rxBuffSize;

    return c;
}

/**
 * An internal method that copies values from a circular buffer to a linear buffer.
 *
 * @param circularBuff a pointer to the source circular buffer
 *
 * @param circularBuffSize the size of the circular buffer
 *
 * @param linearBuff a pointer to the destination linear buffer
 *
 * @param tailPosition the tail position in the circular buffer you want to copy from
 *
 * @param headPosition the head position in the circular buffer you want to copy to
 *
 * @note this method assumes that the linear buffer has the appropriate amount of
 *       memory to contain the copy operation
 */
void pfodSerialUart::circularCopy(uint8_t *circularBuff, uint8_t circularBuffSize, uint8_t *linearBuff, uint16_t tailPosition, uint16_t headPosition) {
    int toBuffIndex = 0;

    while (tailPosition != headPosition) {
        linearBuff[toBuffIndex++] = circularBuff[tailPosition];

        tailPosition = (tailPosition + 1) % circularBuffSize;
    }
}

/**
 * Sends a ManagedString over the serial line.
 *
 * @param s the string to send
 *
 * mode is ALWAYS SYNC_SLEEP - bytes are copied into the txBuff and the fiber sleeps
 *                         until all bytes have been sent. This allows other fibers
 *                         to continue execution.
 *
 * @return the number of bytes written, PFOD_UART_NO_DATA if another fiber
 *         is using the serial instance for transmission, MICROBIT_INVALID_PARAMETER
 *         if buffer is invalid, or the given bufferLen is <= 0.
 */
int pfodSerialUart::send(ManagedString s) {
    return send((uint8_t *) s.toCharArray(), s.length());
}

/**
 * Sends a single character over the serial line.
 *
 * @param c the character to send
 * 
 * the character is copied into the txBuff and the fiber sleeps
 *                         until the character has been sent. This allows other fibers
 *                         to continue execution.
 *
 * @return the number of bytes written, or PFOD_UART_NO_DATA if another fiber
 *         is using the serial instance for transmission.
 */
int pfodSerialUart::sendChar(char c) {
    return write((uint8_t) c);
}

/**
 * Sends a buffer of known length over the serial line.
 *
 * @param buffer a pointer to the first character of the buffer
 *
 * @param len the number of bytes that are safely available to read.
 *
 * mode is ALWAYS SYNC_SLEEP - bytes are copied into the txBuff and the fiber sleeps
 *                         until all bytes have been sent. This allows other fibers
 *                         to continue execution.
 *
 *
 * @return the number of bytes written, PFOD_UART_NO_DATA if another fiber
 *         is using the serial instance for transmission, PFOD_UART_NO_DATA
 *         if buffer is invalid, or the given bufferLen is <= 0.
 */
int pfodSerialUart::send(uint8_t *buffer, int bufferLen) {
    if (txInUse())
        return PFOD_UART_NO_DATA;

    if (bufferLen <= 0 || buffer == NULL)
        return MICROBIT_INVALID_PARAMETER;

    lockTx();

    bool complete = false;
    int bytesWritten = 0;

    while (!complete) {
        bytesWritten += setTxInterrupt(buffer + bytesWritten, bufferLen - bytesWritten);
        send();

        if (bytesWritten >= bufferLen)
            complete = true;
    }

    unlockTx();

    return bytesWritten;
}

/**
 * A wrapper around the inherited method "baud" so we can trap the baud rate
 * as it changes and restore it if redirect() is called.
 *
 * @param baudrate the new baudrate. See:
 *         - https://github.com/mbedmicro/mbed/blob/master/libraries/mbed/targets/hal/TARGET_NORDIC/TARGET_MCU_NRF51822/serial_api.c
 *        for permitted baud rates.
 *
 * @return MICROBIT_INVALID_PARAMETER if baud rate is less than 0, otherwise MICROBIT_OK.
 *
 * @note the underlying implementation chooses the first allowable rate at or above that requested.
 */
void pfodSerialUart::baud(int baudrate) {
    if (baudrate < 0)
        return;

    this->baudrate = baudrate;

    RawSerial::baud(baudrate);
}

/**
 * A way of dynamically configuring the serial instance to use pins other than USBTX and USBRX.
 * clears tx and rx buffers
 *
 * @param tx the new transmission pin.
 *
 * @param rx the new reception pin.
 *
 * @return MICROBIT_SERIAL_IN_USE if another fiber is currently transmitting or receiving, otherwise MICROBIT_OK.
 */
int pfodSerialUart::redirect(PinName tx, PinName rx) {
    if (txInUse() || rxInUse())
        return MICROBIT_SERIAL_IN_USE;

    lockTx();
    lockRx();
    // clear buffers
    rxBuffTail = rxBuffHead;
    txBuffTail = txBuffHead;
    detach(Serial::TxIrq);
    detach(Serial::RxIrq);

    serial_init(&_serial, tx, rx);

    attach(this, &pfodSerialUart::dataReceived, Serial::RxIrq);
    attach(this, &pfodSerialUart::dataWritten, Serial::TxIrq);

    this->baud(this->baudrate);

    unlockRx();
    unlockTx();

    return MICROBIT_OK;
}

/**
 * Determines whether there is any data waiting in our Rx buffer.
 *
 * @return 1 if we have space, 0 if we do not.
 *
 * @note We do not wrap the super's readable() method as we don't want to
 *       interfere with communities that use manual calls to serial.readable().
 */
int pfodSerialUart::isReadable() {
    return (rxBuffTail != rxBuffHead) ? 1 : 0;
}

/**
 * Determines if we have space in our txBuff.
 *
 * @return 1 if we have space, 0 if we do not.
 *
 * @note We do not wrap the super's writeable() method as we don't want to
 *       interfere with communities that use manual calls to serial.writeable().
 */
int pfodSerialUart::isWriteable() {
    return (txBuffHead != (txBuffTail - 1)) ? 1 : 0;
}

/**
 * Reconfigures the size of our rxBuff
 *
 * @param size the new size for our rxBuff
 *
 * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
 *         for reception, otherwise MICROBIT_OK.
 */
int pfodSerialUart::setRxBufferSize(uint8_t size) {
    if (rxInUse())
        return MICROBIT_SERIAL_IN_USE;

    lockRx();

    // + 1 so there is a usable buffer size, of the size the user requested.
    this->rxBuffSize = size + 1;

    int result = initialiseRx();

    unlockRx();

    return result;
}

/**
 * Reconfigures the size of our txBuff
 *
 * @param size the new size for our txBuff
 *
 * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
 *         for transmission, otherwise MICROBIT_OK.
 */
int pfodSerialUart::setTxBufferSize(uint8_t size) {
    if (txInUse())
        return MICROBIT_SERIAL_IN_USE;

    lockTx();

    // + 1 so there is a usable buffer size, of the size the user requested.
    this->txBuffSize = size + 1;

    int result = initialiseTx();

    unlockTx();

    return result;
}

/**
 * The size of our rx buffer in bytes.
 *
 * @return the current size of rxBuff in bytes
 */
int pfodSerialUart::getRxBufferSize() {
    return this->rxBuffSize;
}

/**
 * The size of our tx buffer in bytes.
 *
 * @return the current size of txBuff in bytes
 */
int pfodSerialUart::getTxBufferSize() {
    return this->txBuffSize;
}

/**
 * Sets the tail to match the head of our circular buffer for reception,
 * effectively clearing the reception buffer.
 *
 * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
 *         for reception, otherwise MICROBIT_OK.
 */
int pfodSerialUart::clearRxBuffer() {
    if (rxInUse())
        return MICROBIT_SERIAL_IN_USE;

    lockRx();

    rxBuffTail = rxBuffHead;

    unlockRx();

    return MICROBIT_OK;
}

/**
 * Sets the tail to match the head of our circular buffer for transmission,
 * effectively clearing the transmission buffer.
 *
 * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
 *         for transmission, otherwise MICROBIT_OK.
 */
int pfodSerialUart::clearTxBuffer() {
    if (txInUse())
        return MICROBIT_SERIAL_IN_USE;

    lockTx();

    txBuffTail = txBuffHead;

    unlockTx();

    return MICROBIT_OK;
}

/**
 * Determines if the serial bus is currently in use by another fiber for reception.
 *
 * @return The state of our mutex lock for reception.
 *
 * @note Only one fiber can call read at a time
 */
int pfodSerialUart::rxInUse() {
    return (status & MICROBIT_SERIAL_RX_IN_USE);
}

/**
 * Determines if the serial bus is currently in use by another fiber for transmission.
 *
 * @return The state of our mutex lock for transmition.
 *
 * @note Only one fiber can call send at a time
 */
int pfodSerialUart::txInUse() {
    return (status & MICROBIT_SERIAL_TX_IN_USE);
}

/**
 * Detaches a previously configured interrupt
 *
 * @param interruptType one of Serial::RxIrq or Serial::TxIrq
 */
void pfodSerialUart::detach(Serial::IrqType interruptType) {
    //we detach by sending a bad value to attach, for some weird reason...
    attach((pfodSerialUart *) NULL, &pfodSerialUart::dataReceived, interruptType);
}
