/*
 pfodParserMicroBit library V1.0   20th October 2016 (www.pfod.com.au)
 (c)2016 Forward Computing and Control Pty. Ltd.
 This code is not warranted to be fit for any purpose. You may only use it at your own risk.
 This code may be freely used for both private and commercial use.
 Provide this copyright is maintained.
 */

/*
This code derived from BBC micro:bit C++ MicroBitUARTService.cpp and others and is used under the following licence
The MIT License (MIT)

Copyright (c) 2016 British Broadcasting Corporation.
This software is provided by Lancaster University by arrangement with the BBC.

Permission is hereby granted, free of charge, to any person obtaining a
copy of this software and associated documentation files (the "Software"),
to deal in the Software without restriction, including without limitation
the rights to use, copy, modify, merge, publish, distribute, sublicense,
and/or sell copies of the Software, and to permit persons to whom the
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
 */

#ifndef PFOD_SERIAL_UART_H
#define PFOD_SERIAL_UART_H

#include "mbed.h"
#include "ManagedString.h"

// for MicroBitSerialMode
#include "MicroBitSerial.h" 

#include "pfodParserStream.h"
#include "pfodParser.h"

/**
 * Class definition for the custom MicroBit UART Service.
 * Provides a BLE service that acts as a UART port, enabling the reception and transmission
 * of an arbitrary number of bytes.
 */
class pfodSerialUart : public RawSerial, public pfodParserStream {
public:
    int init(); // initialize buffers

    size_t available(void); // return bytes available for read()

    /**
     * Reads a single character from the rxBuff
     *
     * mode is ALWAYS ASYNC - A character is read from the rxBuff if available, if there
     *                    are no characters to be read, a value of MICROBIT_NO_DATA is returned immediately.
     *
     * so always guard call with available()>0
     *
     * @return a character, MICROBIT_SERIAL_IN_USE if another fiber is using the serial instance for reception,
     *         MICROBIT_NO_RESOURCES if buffer allocation did not complete successfully, or PFOD_UART_NO_DATA if
     *         the rx buffer is empty and the mode given is ASYNC.
     */
    int read(void); // read an input byte from rx buffer. Return PFOD_UART_NO_DATA (-1) if no data

    size_t txFree(void); // return space available for write();
    int write(uint8_t c); // write to tx buffer. Note: spins if no space to write
    void flush(void); // force write of buffered bytes if any
    bool isBle(void); // is this a BLE stream

    /**
     * Constructor.
     * Create an instance of MicroBitSerial
     *
     * @param tx the Pin to be used for transmission
     *
     * @param rx the Pin to be used for receiving data
     *
     * @param rxBufferSize the size of the buffer to be used for receiving bytes
     *
     * @param txBufferSize the size of the buffer to be used for transmitting bytes
     *
     * @code
     * MicroBitSerial serial(USBTX, USBRX);
     * @endcode
     * @note the default baud rate is 115200. More API details can be found:
     *       -https://github.com/mbedmicro/mbed/blob/master/libraries/mbed/api/SerialBase.h
     *       -https://github.com/mbedmicro/mbed/blob/master/libraries/mbed/api/RawSerial.h
     *
     *       Buffers aren't allocated until the first send or receive respectively.
     */
    pfodSerialUart(PinName tx, PinName rx, uint8_t txBufferSize = MICROBIT_SERIAL_DEFAULT_BUFFER_SIZE);


    /**
     * Sends a ManagedString over the serial line.
     *
     * @param s the string to send
     *
     * mode is ALWAYS SYNC_SLEEP - bytes are copied into the txBuff and the fiber sleeps
     *                         until all bytes have been sent. This allows other fibers
     *                         to continue execution.
     *
     * @return the number of bytes written, PFOD_UART_NO_DATA if another fiber
     *         is using the serial instance for transmission, MICROBIT_INVALID_PARAMETER
     *         if buffer is invalid, or the given bufferLen is <= 0.
     */
    int send(ManagedString s);

    /**
      * Sends a single character over the serial line.
      *
      * @param c the character to send
     * 
      * the character is copied into the txBuff and the fiber sleeps
      *                         until the character has been sent. This allows other fibers
      *                         to continue execution.
      *
      * @return the number of bytes written, or PFOD_UART_NO_DATA if another fiber
      *         is using the serial instance for transmission.
      */
    int sendChar( char c);

    /**
     * A wrapper around the inherited method "baud" so we can trap the baud rate
     * as it changes and restore it if redirect() is called.
     *
     * @param baudrate the new baudrate. See:
     *         - https://github.com/mbedmicro/mbed/blob/master/libraries/mbed/targets/hal/TARGET_NORDIC/TARGET_MCU_NRF51822/serial_api.c
     *        for permitted baud rates.
     *
     * @return MICROBIT_INVALID_PARAMETER if baud rate is less than 0, otherwise MICROBIT_OK.
     *
     * @note the underlying implementation chooses the first allowable rate at or above that requested.
     */
    void baud(int baudrate);

    /**
     * A way of dynamically configuring the serial instance to use pins other than USBTX and USBRX.
     *
     * @param tx the new transmission pin.
     *
     * @param rx the new reception pin.
     *
     * @return MICROBIT_SERIAL_IN_USE if another fiber is currently transmitting or receiving, otherwise MICROBIT_OK.
     */
    int redirect(PinName tx, PinName rx);

    /**
     * Sets the tail to match the head of our circular buffer for reception,
     * effectively clearing the reception buffer.
     *
     * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
     *         for reception, otherwise MICROBIT_OK.
     */
    int clearRxBuffer();

    /**
     * Sets the tail to match the head of our circular buffer for transmission,
     * effectively clearing the transmission buffer.
     *
     * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
     *         for transmission, otherwise MICROBIT_OK.
     */
    int clearTxBuffer();




private:
    //holds that state of the mutex locks for all MicroBitSerial instances.
    static uint8_t status;

    //holds the state of the baudrate for all MicroBitSerial instances.
    static int baudrate;

    //delimeters used for matching on receive.
    ManagedString delimeters;

    //a variable used when a user calls the eventAfter() method.
    int rxBuffHeadMatch;

    uint8_t *rxBuff;
    int rxBuffSize;
    volatile uint16_t rxBuffHead;
    uint16_t rxBuffTail;


    uint8_t *txBuff;
    uint8_t txBuffSize;
    uint16_t txBuffHead;
    volatile uint16_t txBuffTail;

private:

/**
 * Sends a buffer of known length over the serial line.
 *
 * @param buffer a pointer to the first character of the buffer
 *
 * @param len the number of bytes that are safely available to read.
 *
 * mode is ALWAYS SYNC_SLEEP - bytes are copied into the txBuff and the fiber sleeps
 *                         until all bytes have been sent. This allows other fibers
 *                         to continue execution.
 *
 *
 * @return the number of bytes written, PFOD_UART_NO_DATA if another fiber
 *         is using the serial instance for transmission, PFOD_UART_NO_DATA
 *         if buffer is invalid, or the given bufferLen is <= 0.
 */
    int send(uint8_t *buffer, int bufferLen);

    /**
     * Determines whether there is any data waiting in our Rx buffer.
     *
     * @return 1 if we have space, 0 if we do not.
     *
     * @note We do not wrap the super's readable() method as we don't want to
     *       interfere with communities that use manual calls to serial.readable().
     */
    int isReadable();

    /**
     * Determines if we have space in our txBuff.
     *
     * @return 1 if we have space, 0 if we do not.
     *
     * @note We do not wrap the super's writeable() method as we don't want to
     *       interfere with communities that use manual calls to serial.writeable().
     */
    int isWriteable();

    /**
     * Determines if the serial bus is currently in use by another fiber for reception.
     *
     * @return The state of our mutex lock for reception.
     *
     * @note Only one fiber can call read at a time
     */
    int rxInUse();

    /**
     * Determines if the serial bus is currently in use by another fiber for transmission.
     *
     * @return The state of our mutex lock for transmition.
     *
     * @note Only one fiber can call send at a time
     */
    int txInUse();

    /**
     * Detaches a previously configured interrupt
     *
     * @param interruptType one of Serial::RxIrq or Serial::TxIrq
     */
    void detach(Serial::IrqType interuptType);

    /**
     * The size of our rx buffer in bytes.
     *
     * @return the current size of rxBuff in bytes
     */
    int getRxBufferSize();

    /**
     * The size of our tx buffer in bytes.
     *
     * @return the current size of txBuff in bytes
     */
    int getTxBufferSize();

    /**
     * Reconfigures the size of our rxBuff
     *
     * @param size the new size for our rxBuff
     *
     * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
     *         for reception, otherwise MICROBIT_OK.
     */
    int setRxBufferSize(uint8_t size);

    /**
     * Reconfigures the size of our txBuff
     *
     * @param size the new size for our txBuff
     *
     * @return MICROBIT_SERIAL_IN_USE if another fiber is currently using this instance
     *         for transmission, otherwise MICROBIT_OK.
     */
    int setTxBufferSize(uint8_t size);

    /**
     * An internal interrupt callback for MicroBitSerial configured for when a
     * character is received.
     *
     * Each time a character is received fill our circular buffer!
     */
    void dataReceived();

    /**
     * An internal interrupt callback for MicroBitSerial.
     *
     * Each time the Serial module's buffer is empty, write a character if we have
     * characters to write.
     */
    void dataWritten();

    /**
     * An internal method to configure an interrupt on tx buffer and also
     * a best effort copy operation to move bytes from a user buffer to our txBuff
     *
     * @param string a pointer to the first character of the users' buffer.
     *
     * @param len the length of the string, and ultimately the maximum number of bytes
     *        that will be copied dependent on the state of txBuff
     *
     *
     * @return the number of bytes copied into the buffer.
     */
    int setTxInterrupt(uint8_t *string, int len);

    /**
     * Locks the mutex so that others can't use this serial instance for reception
     */
    void lockRx();

    /**
     * Locks the mutex so that others can't use this serial instance for transmission
     */
    void lockTx();

    /**
     * Unlocks the mutex so that others can use this serial instance for reception
     */
    void unlockRx();

    /**
     * Unlocks the mutex so that others can use this serial instance for transmission
     */
    void unlockTx();

    /**
     * We do not want to always have our buffers initialised, especially if users to not
     * use them. We only bring them up on demand.
     */
    int initialiseRx();

    /**
     * We do not want to always have our buffers initialised, especially if users to not
     * use them. We only bring them up on demand.
     */
    int initialiseTx();

    /**
     * An internal method that puts the fiber to sleep
     *
     */
    void send();

    /**
     * Reads a single character from the rxBuff
     * if there are no characters to be read, a value of PFOD_UART_NO_DATA is returned immediately.
     *
     * @return a character from the circular buffer, or PFOD_UART_NO_DATA is there
     *         are no characters in the buffer.
     */
    int getChar(void);

    /**
     * An internal method that copies values from a circular buffer to a linear buffer.
     *
     * @param circularBuff a pointer to the source circular buffer
     *
     * @param circularBuffSize the size of the circular buffer
     *
     * @param linearBuff a pointer to the destination linear buffer
     *
     * @param tailPosition the tail position in the circular buffer you want to copy from
     *
     * @param headPosition the head position in the circular buffer you want to copy to
     *
     * @note this method assumes that the linear buffer has the appropriate amount of
     *       memory to contain the copy operation
     */
    void circularCopy(uint8_t *circularBuff, uint8_t circularBuffSize, uint8_t *linearBuff, uint16_t tailPosition, uint16_t headPosition);

};

#endif
